import {useCallback, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import {IResultAviaStation} from 'selectors/avia/utils/denormalization/station';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {IAviaCityDirectionGroup} from 'projects/avia/components/AviaSearchResultsTransferFilter/AviaTransferFilterPopup/components/AviaCityDirectionGroup/helpers/preparedCities';

import * as i18nBlock from 'i18n/avia-search';

import Checkbox from 'components/Checkbox/Checkbox';
import {
    AviaCityCheckbox,
    ICityDirectionGroups,
} from 'projects/avia/components/AviaSearchResultsTransferFilter/AviaTransferFilterPopup/components/AviaCityCheckbox/AviaCityCheckbox';
import Flex from 'components/Flex/Flex';

interface IAviaCityDirectionGroupProps extends IWithClassName {
    cities: IAviaCityDirectionGroup[];
    stations: IResultAviaStation[];
    segmentIdx: number;
    segmentFilter: number[] | undefined;
    disabledStations?: Record<number, boolean>;
    onChangeDiff: (
        stationIds: number[],
        segmentIdx: number,
        checked: boolean,
    ) => void;
    onChange: (stationIds: number[], segmentIdx: number) => void;
    transfersIsDisabled: boolean;
}

export function AviaCityDirectionGroup({
    className,
    cities,
    stations,
    segmentIdx,
    segmentFilter,
    disabledStations = {},
    onChangeDiff,
    onChange,
    transfersIsDisabled,
}: IAviaCityDirectionGroupProps): JSX.Element | null {
    const deviceType = useDeviceType();
    const {isMobile} = deviceType;
    const handleChangeDiff = useCallback(
        (checked: boolean, city: ICityDirectionGroups) => {
            const stationIds = city.stations.map(station => station.id);

            onChangeDiff(stationIds, segmentIdx, checked);
        },
        [onChangeDiff, segmentIdx],
    );
    const handleChange = useCallback(
        (newStations: IResultAviaStation[]) => {
            const stationIds = newStations.map(station => station.id);

            onChange(stationIds, segmentIdx);
        },
        [onChange, segmentIdx],
    );
    const allCitiesIsChecked = useMemo(
        () => segmentFilter?.length === stations.length,
        [segmentFilter, stations],
    );
    const handleAllCitiesChange = useCallback(() => {
        const selectedStations = allCitiesIsChecked ? [] : stations;

        handleChange(selectedStations);
    }, [allCitiesIsChecked, handleChange, stations]);

    const renderCities = useMemo(
        () =>
            cities.map((city: ICityDirectionGroups) => {
                const isChecked = segmentFilter
                    ? segmentFilter.includes(city.stations[0].id)
                    : false;
                const disabled =
                    transfersIsDisabled ||
                    city.stations
                        .map(station => station.id)
                        .map(id => disabledStations[id])
                        .every(Boolean);

                return (
                    <AviaCityCheckbox
                        checked={isChecked}
                        disabled={disabled}
                        onChangeDiff={handleChangeDiff}
                        onChange={handleChange}
                        city={city}
                        key={city.stations[0].id}
                        transfersIsDisabled={transfersIsDisabled}
                    />
                );
            }),
        [
            cities,
            disabledStations,
            handleChange,
            handleChangeDiff,
            segmentFilter,
            transfersIsDisabled,
        ],
    );

    if (cities.length === 0) {
        return null;
    }

    return (
        <div className={className}>
            <Checkbox
                checked={allCitiesIsChecked}
                disabled={transfersIsDisabled}
                onChange={handleAllCitiesChange}
                label={i18nBlock.filterDotTransfersDotAllDashCities()}
            />
            <Flex
                above={isMobile ? 8 : 6}
                between={isMobile ? 6 : 3}
                flexDirection="column"
            >
                {renderCities}
            </Flex>
        </div>
    );
}
