import React from 'react';

import {MINUTE} from 'utilities/dateUtils/constants';

import {EAviaSearchTransferFilter} from 'reducers/avia/search/results/filters/reducer';

import {Flag, isEnabled} from 'utilities/flags/flags';
import {tupple} from 'projects/avia/lib/tupple';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/avia-search';

import Box from 'components/Box/Box';
import InputRange from 'components/InputRange/InputRange';
import {Duration} from 'projects/avia/components/Duration/Duration';
import Text from 'components/Text/Text';

import cx from './AviaRange.scss';

interface IAviaRangeProps {
    value: Flag<EAviaSearchTransferFilter>;
    range?: [number, number] | null;
    minDuration: number;
    maxDuration: number;
    onRangeChange?: (range: [number, number]) => void;
}

export function AviaRange({
    value,
    range,
    minDuration,
    maxDuration,
    onRangeChange,
}: IAviaRangeProps) {
    const transfersIsDisabled = isEnabled(
        value,
        EAviaSearchTransferFilter.NO_TRANSFERS,
    );

    const handleRangeChange = React.useCallback(
        (nextRange: Readonly<number[]>) => {
            onRangeChange?.([nextRange[0], nextRange[1]]);
        },
        [onRangeChange],
    );

    const priceRange = tupple(minDuration, maxDuration);
    const {isMobile} = useDeviceType();

    return (
        <Box above="8">
            <Box below="5">
                <Text weight="bold" disabled={transfersIsDisabled}>
                    {i18nBlock.filterDotTransfersDotTransferDashDuration()}
                </Text>
            </Box>
            <div
                className={cx('range-container', {
                    'range-container_mobile': isMobile,
                })}
            >
                <InputRange
                    disabled={transfersIsDisabled}
                    rangeValues={priceRange}
                    values={range || priceRange}
                    renderHandleTooltip={props => (
                        <Duration
                            className={cx('range-tooltip')}
                            duration={props.value}
                            withoutOffset
                            disabled={transfersIsDisabled}
                        />
                    )}
                    size={isMobile ? 'm' : 's'}
                    tooltipPosition="top"
                    step={5 * MINUTE}
                    onChange={handleRangeChange}
                />
            </div>
        </Box>
    );
}
