import React, {useCallback, useState} from 'react';
import {connect, useSelector} from 'react-redux';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {IWithClassName} from 'src/types/withClassName';

import {DispatchProps} from 'reducers/trains/customDispatch';
import {unsubscribe} from 'reducers/avia/search/subscription/thunk';
import {setFavorite, unsetFavorite} from 'reducers/avia/platform/actions';

import {aviaSubscriptionSelector} from 'selectors/avia/search/subscription';
import {getGroupedVariantsInfo} from 'selectors/avia/search/getGroupedVariantsInfo';
import {getAviaMinSearchPrice} from 'selectors/avia/search/getMinSearchPrice';

import {
    useCoordinator,
    usePlatform,
    INotifierAddToFavorites,
} from 'contexts/PlatformContext';

import AviaSearchSubscribeModal from './AviaSearchSubscribeModal/AviaSearchSubscribeModal';

export interface IAviaSubscriptionRenderProps {
    subscribed: boolean;
    unsubscribe: () => void;
    toggleModal: () => void;
}

interface IOwnProps extends IWithClassName {
    renderTrigger: (
        renderProps: IAviaSubscriptionRenderProps,
    ) => React.ReactNode;
}

interface IAviaSearchSubscribeProps
    extends IOwnProps,
        DispatchProps,
        ReturnType<typeof aviaSubscriptionSelector> {}

const AviaSearchSubscribe: React.FC<IAviaSearchSubscribeProps> = ({
    renderTrigger,
    subscription,
    dispatch,
}) => {
    const {nativeAppState} = usePlatform();
    const {id, needConfirmation} = subscription;
    const isSubscribed =
        nativeAppState.isFavorite || (Boolean(id) && !needConfirmation);
    const [isModalVisible, setModalIsVisible] = useState(false);
    const coordinator = useCoordinator();
    const {all: allVariantsCount} = useSelector(getGroupedVariantsInfo);
    const minPrice = useSelector(getAviaMinSearchPrice);

    const unsubscribeHandler = useCallback(() => {
        const handled = coordinator.doAction(EAppActions.REMOVE_FROM_FAVORITES);

        if (handled) {
            dispatch(unsetFavorite());

            return;
        }

        dispatch(unsubscribe(id));
    }, [dispatch, id, coordinator]);

    const modalStateHandler = useCallback(() => {
        const handled = coordinator.doAction<INotifierAddToFavorites>(
            EAppActions.ADD_TO_FAVORITES,
            {
                offersCount: allVariantsCount,
                minTariff: minPrice,
            },
        );

        if (handled) {
            dispatch(setFavorite());

            return;
        }

        setModalIsVisible(!isModalVisible);
    }, [coordinator, allVariantsCount, isModalVisible, dispatch, minPrice]);

    return (
        <>
            {renderTrigger({
                subscribed: isSubscribed,
                toggleModal: modalStateHandler,
                unsubscribe: unsubscribeHandler,
            })}

            <AviaSearchSubscribeModal
                isVisible={isModalVisible}
                onClose={modalStateHandler}
            />
        </>
    );
};

export default connect(aviaSubscriptionSelector)(AviaSearchSubscribe);
