import React from 'react';

import {IWithClassName} from 'src/types/withClassName';

import {IResultVariantPrice} from 'selectors/avia/utils/denormalization/prices';

import {deviceModMobile} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useCurrencyConverter} from 'utilities/hooks/useCurrencyConverter';

import Flex, {IFlexProps} from 'components/Flex/Flex';
import Price from 'components/Price/Price';

import cx from './AviaVariantPrice.scss';

interface IProps
    extends IWithClassName,
        IWithQaAttributes,
        Pick<IFlexProps, 'flexDirection' | 'alignItems'> {
    nationalPriceClassName?: string;
    originPriceClassName?: string;
    price: IResultVariantPrice;
    theme: 'search' | 'order';
}

const AviaVariantPrice: React.FC<IProps> = ({
    className,
    nationalPriceClassName,
    price,
    theme,
    alignItems = 'flex-end',
    flexDirection = 'column',
    ...rest
}) => {
    const deviceType = useDeviceType();
    const priceConverter = useCurrencyConverter();

    const isPreferredCurrency = priceConverter.isPreferredCurrency(
        price.tariff.currency,
    );

    return (
        <Flex
            className={cx(
                'root',
                deviceModMobile('root', deviceType),
                `root_theme_${theme}`,
                className,
            )}
            alignItems={alignItems}
            flexDirection={flexDirection}
        >
            <Price
                withConversion={!isPreferredCurrency}
                isRoughly={!isPreferredCurrency}
                isRound={!isPreferredCurrency}
                className={cx(
                    'price',
                    'price_national',
                    nationalPriceClassName,
                )}
                {...price.tariff}
                {...prepareQaAttributes(rest)}
            />
        </Flex>
    );
};

export default AviaVariantPrice;
