import React from 'react';
import {upperFirst} from 'lodash';

import {IWithClassName} from 'src/types/withClassName';

import {IResultVariantPrice} from 'selectors/avia/utils/denormalization/prices';

import {deviceModMobile} from 'utilities/stylesUtils';
import {useToggle} from 'utilities/hooks/useToggle';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useCurrencyConverter} from 'utilities/hooks/useCurrencyConverter';

import * as i18nBlock from 'i18n/avia_disclaimer';
import * as i18nCommonDisclaimerBlock from 'i18n/common_disclaimer';

import Flex, {IFlexProps} from 'components/Flex/Flex';
import Modal from 'components/Modal/Modal';
import Price from 'components/Price/Price';
import Button from 'components/Button/Button';
import InfoIcon from 'icons/16/Info';

import cx from './AviaVariantPrice.scss';

interface IProps
    extends IWithClassName,
        IWithQaAttributes,
        Pick<IFlexProps, 'flexDirection' | 'alignItems'> {
    nationalPriceClassName?: string;
    originPriceClassName?: string;
    price: IResultVariantPrice;
    theme: 'search' | 'order';
}

const AviaVariantPriceWithHint: React.FC<IProps> = props => {
    const {
        className,
        originPriceClassName,
        nationalPriceClassName,
        price,
        theme,
        alignItems = 'flex-end',
        flexDirection = 'column',
    } = props;
    const deviceType = useDeviceType();
    const priceConverter = useCurrencyConverter();
    const [hintIsVisible, toggleHint] = useToggle(false);

    const {isMobile} = deviceType;
    const iconSize = theme === 'search' ? 14 : 12;
    const isPreferredCurrency = priceConverter.isPreferredCurrency(
        price.tariff.currency,
    );

    return (
        <Flex
            className={cx(
                'root',
                deviceModMobile('root', deviceType),
                `root_theme_${theme}`,
                className,
            )}
            alignItems={alignItems}
            flexDirection={flexDirection}
        >
            <Price
                withConversion={!isPreferredCurrency}
                isRoughly={!isPreferredCurrency}
                isRound={!isPreferredCurrency}
                className={cx(
                    'price',
                    'price_national',
                    nationalPriceClassName,
                )}
                {...price.tariff}
                {...prepareQaAttributes(props)}
            />

            {!isPreferredCurrency && (
                <Flex alignItems="center" inline between={1}>
                    <Price
                        className={cx(
                            'price',
                            'price_origin',
                            originPriceClassName,
                        )}
                        withConversion={false}
                        {...price.tariff}
                    />

                    <span className={cx('iconWrapper')} onClick={toggleHint}>
                        <InfoIcon
                            width={iconSize}
                            height={iconSize}
                            fill="currentColor"
                        />
                    </span>

                    <Modal
                        isMobile={isMobile}
                        onClose={toggleHint}
                        isVisible={hintIsVisible}
                        preventBodyScroll={false}
                    >
                        <Modal.Content className={cx('content')}>
                            <div className={cx('message')}>
                                {i18nBlock.ruAviaCurrencyNotNationalContent()}
                            </div>

                            <Button theme="primary" onClick={toggleHint}>
                                {upperFirst(i18nCommonDisclaimerBlock.ok())}
                            </Button>
                        </Modal.Content>
                    </Modal>
                </Flex>
            )}
        </Flex>
    );
};

export default AviaVariantPriceWithHint;
