import React, {ReactNode} from 'react';

import {IWithClassName} from 'types/withClassName';

import {getDuration} from 'projects/avia/lib/date/getDuration';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/avia-Duration';

import Text from 'components/Text/Text';

import cx from './Duration.scss';

interface DurationProps extends IWithClassName, IWithQaAttributes {
    duration: number | {get(): number};
    view?: 'search' | 'booking' | 'order';
    withoutOffset?: boolean;
    disabled?: boolean;
}

const SECOND = 1000;
const MINUTE = 60 * SECOND;
const HOUR = 60 * MINUTE;

export class Duration extends React.Component<DurationProps> {
    static milliseconds(ms: number) {
        return duration(ms);
    }

    static seconds(s: number) {
        return duration().seconds(s);
    }

    static minutes(m: number) {
        return duration().minutes(m);
    }

    static hours(h: number) {
        return duration().hours(h);
    }

    render(): ReactNode {
        // eslint-disable-next-line no-shadow
        const {duration, view, className, withoutOffset, disabled} = this.props;
        const milliseconds =
            typeof duration === 'number' ? duration : duration.get();
        const content = i18nBlock
            .dDashHDashMDashFormat(getDuration(milliseconds))
            .trim();

        return (
            <Text
                className={cx(
                    'root',
                    {
                        [`root_view_${view}`]: Boolean(view),
                        root_withoutOffset: withoutOffset,
                    },
                    className,
                )}
                disabled={disabled}
                {...prepareQaAttributes(this.props)}
            >
                <div className={cx('dash')} />
                <div className={cx('content')}>{content}</div>
                <div className={cx('dash')} />
            </Text>
        );
    }
}

function duration(acc: number = 0) {
    return {
        milliseconds(ms: number) {
            return duration(acc + ms);
        },
        seconds(s: number) {
            return duration(acc + s * SECOND);
        },
        minutes(m: number) {
            return duration(acc + m * MINUTE);
        },
        hours(h: number) {
            return duration(acc + h * HOUR);
        },
        get() {
            return acc;
        },
    };
}

export default Duration;
