import React from 'react';
import moment from 'moment';
import upperFirst from 'lodash/upperFirst';

import {IWithClassName} from 'types/withClassName';
import {IDynamicsCalendarDay} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import {TInlineSearchRequest} from 'projects/avia/components/Dynamics/types/TInlineSearchRequest';

import {IAviaContext} from 'reducers/avia/data-types';

import isCurrentDynamicDay from 'projects/avia/components/Dynamics/utilities/isCurrentDynamicDay';

import DayButton from './components/DayButton/DayButton';

import cx from './Month.scss';

interface IMonthProps extends IWithClassName {
    aviaContext: IAviaContext;
    days: IDynamicsCalendarDay[];
    inlineSearchRequest: TInlineSearchRequest;
    isCurrentDayHightlighted: boolean;

    onLinkClick?: () => void; // Колбэк на клик по ссылке на поиск
}

function Month({
    aviaContext,
    days,
    inlineSearchRequest,
    isCurrentDayHightlighted,
    onLinkClick,
    className,
}: IMonthProps): React.ReactElement | null {
    if (!days.length) {
        return null;
    }

    const dayMoment = moment(days[0].dateForward.date);

    return (
        <div className={cx('root', className)}>
            <div className={cx('month')} key={dayMoment.format('YYYY-MM')}>
                {upperFirst(dayMoment.format('MMMM'))}
            </div>

            {days.map(day => (
                <DayButton
                    key={`${day.dateForward.date}-${day.dateBackward?.date}`}
                    className={cx('day')}
                    status={day.status}
                    url={day.url}
                    dateForward={day.dateForward}
                    dateBackward={day.dateBackward}
                    type={day.type}
                    price={day.price}
                    isCurrent={
                        isCurrentDayHightlighted
                            ? isCurrentDynamicDay(aviaContext, day)
                            : false
                    }
                    onLinkClick={onLinkClick}
                    inlineSearchRequest={inlineSearchRequest}
                />
            ))}
        </div>
    );
}

export default React.memo(Month);
