import React from 'react';
import moment from 'moment';

import {IWithClassName} from 'types/withClassName';
import {EDynamicsDayStatus} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import {TInlineSearchRequest} from 'projects/avia/components/Dynamics/types/TInlineSearchRequest';
import {IAviaPriceIndexPrice} from 'server/api/AviaPriceIndexApi/types/IAviaPriceIndexPrice';
import ICalendarDate from 'types/common/date/ICalendarDate';

import {EPriceHintType} from 'projects/avia/components/Dynamics/utilities/getPriceHintType';
import {CHAR_MINUS} from 'utilities/strings/charCodes';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';

import {
    dynamicsDashDayDashNoDashFlights,
    dynamicsDashError,
    dynamicsDashUnknownDashPrice,
} from 'i18n/avia-AviaDynamics';

import Price from 'components/Price/Price';
import Spinner from 'components/Spinner/Spinner';
import CloseForCircleBgRefresh16 from 'icons/24/CloseForCircleBgRefresh16';

import cx from './DayButton.scss';

interface IDayButtonProps extends IWithClassName {
    type: EPriceHintType;
    url: string;
    dateForward: ICalendarDate;
    dateBackward: Nullable<ICalendarDate>;
    status: EDynamicsDayStatus;
    price: Nullable<IAviaPriceIndexPrice>;
    inlineSearchRequest: TInlineSearchRequest;
    isCurrent: boolean; // Признак что день является датой "туда" из поиска

    onLinkClick?: () => void; // Колбэк на клик по ссылке на поиск
}

const DayButton: React.FC<IDayButtonProps> = ({
    className,
    type,
    url,
    dateForward,
    dateBackward,
    status,
    price,
    inlineSearchRequest,
    isCurrent,
    onLinkClick,
}) => {
    const forwardMoment = moment(dateForward.date);
    const forwardDay = forwardMoment.format('D');
    const forwardWeekDay = forwardMoment.format('dd');

    const backwardMoment = dateBackward ? moment(dateBackward.date) : undefined;
    const backwardDay = backwardMoment?.format('D');
    const backwardWeekDay = backwardMoment?.format('dd');
    const backwardMonth =
        backwardMoment &&
        backwardMoment.format('M') !== forwardMoment.format('M')
            ? backwardMoment.format('MMM')
            : undefined;

    const isMinimalPrice = type === EPriceHintType.MINIMAL;
    const isSearchLink = !isCurrent && price && !price.roughly;
    const Tag = isSearchLink ? 'a' : 'div';

    const clickHandler = useImmutableCallback(() => {
        if (
            status !== EDynamicsDayStatus.NO_DATA &&
            status !== EDynamicsDayStatus.SEARCHING
        ) {
            inlineSearchRequest(dateForward.date, dateBackward?.date || null);
        }
    });

    let rightContent: React.ReactElement | string;

    switch (status) {
        case EDynamicsDayStatus.NO_DATA:
            rightContent = dynamicsDashDayDashNoDashFlights();

            break;
        case EDynamicsDayStatus.HAS_DATA:
        case EDynamicsDayStatus.SHOULD_SEARCH:
            rightContent = price ? (
                <Price
                    {...price}
                    prefix={price.roughly ? '≈' : undefined}
                    prefixDelimiter=" "
                    className={cx('price')}
                />
            ) : (
                <>
                    {dynamicsDashUnknownDashPrice()}
                    <CloseForCircleBgRefresh16 className={cx('refreshIcon')} />
                </>
            );

            break;
        case EDynamicsDayStatus.SEARCHING:
            rightContent = <Spinner size="xs" />;

            break;
        case EDynamicsDayStatus.WAS_ERROR:
            rightContent = (
                <>
                    {dynamicsDashError()}
                    <CloseForCircleBgRefresh16 className={cx('refreshIcon')} />
                </>
            );

            break;
    }

    return (
        <Tag
            className={cx('root', className, {
                root_isCurrent: isCurrent,
                root_isMinimalPrice: isMinimalPrice,
                root_isError: status === EDynamicsDayStatus.WAS_ERROR,
                root_isNoData: status === EDynamicsDayStatus.NO_DATA,
            })}
            href={isSearchLink ? url : undefined}
            onClick={isSearchLink ? onLinkClick : clickHandler}
            target="_blank"
        >
            <span className={cx('left')}>
                {forwardDay}{' '}
                {dateForward?.isHoliday ? (
                    <span className={cx('holiday')}>{forwardWeekDay}</span>
                ) : (
                    forwardWeekDay
                )}
                {backwardMoment && ` ${CHAR_MINUS} `}
                {backwardDay}{' '}
                {dateBackward?.isHoliday ? (
                    <span className={cx('holiday')}>{backwardWeekDay}</span>
                ) : (
                    backwardWeekDay
                )}{' '}
                {backwardMonth && (
                    <span className={cx('month')}>{backwardMonth}</span>
                )}
            </span>

            <span className={cx('right')}>{rightContent}</span>
        </Tag>
    );
};

export default React.memo(DayButton);
