import React, {ReactNode, useCallback} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import EAviaDynamicActionLogActionName from 'server/loggers/avia/AviaActionLog/types/EAviaDynamicActionLogActionName';

import {EAviaSearchTransferFilter} from 'reducers/avia/search/results/filters/reducer';
import {loggerActions} from 'reducers/avia/aviaLogging/actions';

import {aviaContextSelector} from 'selectors/avia/context/aviaContextSelector';
import {aviaTimeFilterSelector} from 'selectors/avia/search/filters/timeFilter';
import {aviaMobileDynamicsActiveFiltersCountSelector} from 'selectors/avia/search/filters/activeFilters';

import {isEnabled} from 'utilities/flags/flags';
import {isValidDate} from 'utilities/dateUtils';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useAviaBaggageFilter} from 'projects/avia/lib/hooks/filters/baggage';
import {useAviaNoTransfersFilter} from 'projects/avia/lib/hooks/filters/noTransfers';

import * as i18nBlock from 'i18n/avia-search';

import Box from 'components/Box/Box';
import FilterIcon from 'icons/16/Filter';
import Flex from 'components/Flex/Flex';
import CheckButton from 'components/CheckButton/CheckButton';
import Raskrivashka from 'components/Raskrivashka/Raskrivashka';
import {AviaTransferFilterPopupConnected} from 'projects/avia/components/AviaSearchResultsTransferFilter/AviaTransferFilterPopupConnected';
import {AviaSearchTimeFilterConnected} from 'projects/avia/components/AviaSearchResultsTimeFilter/AviaSearchTimeFilterConnected';
import BottomSheet from 'components/BottomSheet/BottomSheet';
import CornerCounter from 'components/CornerCounter/CornerCounter';

import cx from './AviaDynamicsCalendarFilters.scss';

interface IAviaDynamicCalendarFilters {}

const AviaDynamicsCalendarFilters: React.FC<IAviaDynamicCalendarFilters> =
    () => {
        const {
            value: filtersModalIsVisible,
            setFalse: closeFiltersModal,
            setTrue: openFiltersModal,
        } = useBoolean(false);
        const dispatch = useDispatch();
        const context = useSelector(aviaContextSelector);
        const isRoundtrip =
            context.return_date && isValidDate(context.return_date);
        const {filter: transfersFilter, handle: handleTransferChange} =
            useAviaNoTransfersFilter();
        const {filter: baggageFilter, handle: handleBaggageChange} =
            useAviaBaggageFilter();
        const activeFiltersCount = useSelector(
            aviaMobileDynamicsActiveFiltersCountSelector,
        );

        const timeFilterIsActive = useSelector(aviaTimeFilterSelector.isActive);
        const {
            value: {enabled: baggage},
        } = baggageFilter;
        const {
            value: {value: flags},
            isActive: transfersFilterIsActive,
        } = transfersFilter;
        const transfers = isEnabled(
            flags,
            EAviaSearchTransferFilter.NO_TRANSFERS,
        );

        const onClickTransfer = useCallback(() => {
            handleTransferChange();
            dispatch(
                loggerActions.logDynamicAction({
                    name: transfers
                        ? EAviaDynamicActionLogActionName.DYNAMIC_NO_TRANSFERS_OFF
                        : EAviaDynamicActionLogActionName.DYNAMIC_NO_TRANSFERS_ON,
                }),
            );
        }, [dispatch, handleTransferChange, transfers]);

        const onClickBaggage = useCallback(() => {
            handleBaggageChange();
            dispatch(
                loggerActions.logDynamicAction({
                    name: baggage
                        ? EAviaDynamicActionLogActionName.DYNAMIC_BAGGAGE_OFF
                        : EAviaDynamicActionLogActionName.DYNAMIC_BAGGAGE_ON,
                }),
            );
        }, [baggage, dispatch, handleBaggageChange]);

        const onClickTransferRaskrivashka = useCallback(() => {
            dispatch(
                loggerActions.logDynamicAction({
                    name: EAviaDynamicActionLogActionName.DYNAMIC_TRANSFER_FILTER_CLICK,
                }),
            );
        }, [dispatch]);

        const onChangeTransferFilter = useCallback(() => {
            dispatch(
                loggerActions.logDynamicAction({
                    name: EAviaDynamicActionLogActionName.DYNAMIC_TRANSFER_FILTER,
                }),
            );
        }, [dispatch]);

        const onClickTimeRaskrivashka = useCallback(() => {
            dispatch(
                loggerActions.logDynamicAction({
                    name: EAviaDynamicActionLogActionName.DYNAMIC_TIME_FILTER_CLICK,
                }),
            );
        }, [dispatch]);

        const onChangeTimeFilter = useCallback(() => {
            dispatch(
                loggerActions.logDynamicAction({
                    name: EAviaDynamicActionLogActionName.DYNAMIC_TIME_FILTER,
                }),
            );
        }, [dispatch]);

        const filters = (
            <>
                <Raskrivashka
                    contentIsInvolved={transfersFilterIsActive}
                    label={i18nBlock.filterDotTransfersDashPlaceholder()}
                    onClickButton={onClickTransferRaskrivashka}
                    initialOpened
                >
                    <Box x="4">
                        <AviaTransferFilterPopupConnected
                            onChangeCheckbox={onChangeTransferFilter}
                            hideRange
                            hideTransferAirports
                        />
                    </Box>
                </Raskrivashka>

                <Raskrivashka
                    contentIsInvolved={timeFilterIsActive}
                    label={i18nBlock.filterDotTimeDotTitle()}
                    onClickButton={onClickTimeRaskrivashka}
                    initialOpened
                >
                    <Box x="4" between="5">
                        <AviaSearchTimeFilterConnected
                            isRoundtrip={Boolean(isRoundtrip)}
                            wrapFilter={renderTimeFilter}
                            onChange={onChangeTimeFilter}
                        />
                    </Box>
                </Raskrivashka>
            </>
        );

        return (
            <Flex inline nowrap between="2">
                <CheckButton
                    width="max"
                    size="l"
                    checked={transfers}
                    onClick={onClickTransfer}
                >
                    <span className={cx('button-text')}>
                        {i18nBlock.filterDotNoDashTransfers()}
                    </span>
                </CheckButton>

                <CheckButton
                    width="max"
                    size="l"
                    checked={baggage}
                    onClick={onClickBaggage}
                >
                    <span className={cx('button-text')}>
                        {i18nBlock.filterDotWithDashBaggage()}
                    </span>
                </CheckButton>

                <CornerCounter count={activeFiltersCount}>
                    <CheckButton
                        icon={<FilterIcon />}
                        checked={activeFiltersCount > 0}
                        onClick={openFiltersModal}
                        size="l"
                    />
                </CornerCounter>

                <BottomSheet
                    isOpened={filtersModalIsVisible}
                    onClose={closeFiltersModal}
                    contentClassName={cx('modal-container')}
                    maxSize="calc(100% - 68px)"
                >
                    {filters}
                </BottomSheet>
            </Flex>
        );
    };

function renderTimeFilter(label: string, filter: ReactNode): ReactNode {
    return (
        <div>
            <Box below="3" textSize="m" textWeight="bold">
                {label}
            </Box>
            {filter}
        </div>
    );
}

export default AviaDynamicsCalendarFilters;
