import React, {useCallback, useEffect, useContext, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useHistory} from 'react-router-dom';

import EAsyncStatus from 'types/common/EAsyncStatus';
import {
    IDynamicsDay,
    EDynamicsDaysMode,
} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import TValidDataForRequestDynamic from 'types/avia/dynamic/TValidDataForRequestDynamic';

import {dynamicsRequest} from 'reducers/avia/aviaPriceIndex/dynamics/dynamicsRequest';
import {
    getInitialState,
    IAviaSearchResultsFilters,
} from 'reducers/avia/search/results/filters/reducer';
import additionalDynamicRequest from 'reducers/avia/aviaPriceIndex/dynamics/additionalDynamicRequest';

import {aviaDynamicsPricesFinalPriceSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesFinalPriceSelector';
import {
    aviaDynamicsPricesIntervalSelector,
    aviaDynamicsPricesStatusSelector,
} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesSelectors';
import {aviaDynamicsPricesSearchFormSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesSearchFormSelector';
import {aviaDynamicsPricesFiltersSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesFiltersSelector';
import aviaDynamicsPricesCurrentRequestParamsSelector from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesCurrentRequestParamsSelector';

import getDynamicInfo from 'projects/avia/lib/dynamic/getDynamicInfo';
import {hashToFilterValues} from 'projects/avia/lib/search/filters/converters';
import {getAviaDynamicsDesktopInterval} from 'projects/avia/components/Dynamics/utilities/getAviaDynamicsDesktopInterval';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {TInterval} from 'projects/avia/components/Dynamics/utilities/getAviaDynamicsCalendarDates';
import isDateRobot from 'utilities/dateUtils/isDateRobot';

import * as i18nBlock from 'i18n/avia-AviaDynamics';

import Button from 'components/Button/Button';
import Spinner from 'components/Spinner/Spinner';
import ChartSearchSuccess from './components/ChartSearchSuccess/ChartSearchSuccess';
import {TAddData} from 'projects/avia/components/Dynamics/Chart/DynamicsChart/components/DynamicsDays/DynamicsDays';

import {PriceComparatorContext} from 'contexts/PriceComparatorContext';

import cx from './ChartDynamic.scss';

const FILTERS_DEFAULT = getInitialState();

export type TView = 'modal' | 'yak';

interface IChartDynamicProps {
    standalone: boolean;
    mode: EDynamicsDaysMode;
    selectedDate: Nullable<string>;
    onDaysScroll?: () => void;
    onDaySelect: (date: string) => void;
    onLinkClick: (
        date: IDynamicsDay,
        event: React.MouseEvent<HTMLLinkElement>,
    ) => void;

    view?: TView;
}

const ChartDynamic: React.FC<IChartDynamicProps> = ({
    standalone,
    mode,
    selectedDate,
    onDaysScroll,
    onDaySelect,
    onLinkClick,

    view,
}) => {
    const history = useHistory();
    const dispatch = useDispatch();
    const priceComparator = useContext(PriceComparatorContext);

    const hash = history.location.hash.slice(1);
    const filtersForStandalone: IAviaSearchResultsFilters = useMemo(
        () => ({...FILTERS_DEFAULT, ...hashToFilterValues(hash)}),
        [hash],
    );
    const filtersForConnected = useSelector(aviaDynamicsPricesFiltersSelector);

    const searchForm = useSelector(aviaDynamicsPricesSearchFormSelector);
    const pricesFromState = useSelector(aviaDynamicsPricesFinalPriceSelector);
    const currentRequestParams = useSelector(
        aviaDynamicsPricesCurrentRequestParamsSelector,
    );
    const status = useSelector(aviaDynamicsPricesStatusSelector);

    const intervalFromState = useSelector(aviaDynamicsPricesIntervalSelector);
    const initialInterval = getAviaDynamicsDesktopInterval(searchForm);
    const interval =
        intervalFromState &&
        initialInterval &&
        intervalFromState.startDate <= initialInterval.startDate &&
        intervalFromState.endDate >= initialInterval.endDate
            ? intervalFromState
            : initialInterval;

    const {needRequest, actualStatus, requestParams, prices} = getDynamicInfo({
        requestParams: interval
            ? {
                  searchForm,
                  filters: standalone
                      ? filtersForStandalone
                      : filtersForConnected,
                  interval,
              }
            : null,
        currentRequestParams,
        status,
        prices: pricesFromState,
    });

    const searchIsInProcess = actualStatus === EAsyncStatus.LOADING;
    const searchIsFailed = actualStatus === EAsyncStatus.ERROR;
    const searchIsSuccessful = actualStatus === EAsyncStatus.SUCCESS;

    const loadData = useCallback(() => {
        if (requestParams) {
            dispatch(dynamicsRequest(requestParams));
        }
    }, [dispatch, requestParams]);

    const addData: TAddData = useImmutableCallback((startDate, endDate) => {
        if (!isDateRobot(startDate) || !isDateRobot(endDate)) {
            return;
        }

        const newInterval: TInterval = {
            startDate:
                interval && interval.startDate <= startDate
                    ? interval.startDate
                    : startDate,
            endDate:
                interval && interval.endDate >= endDate
                    ? interval.endDate
                    : endDate,
        };

        const newRequestParams: TValidDataForRequestDynamic | undefined =
            requestParams && {
                ...requestParams,
                interval: newInterval,
            };

        if (newRequestParams) {
            dispatch(additionalDynamicRequest(newRequestParams));
        }
    });

    const onRetryLoad = useImmutableCallback(() => {
        loadData();
    });

    useEffect(() => {
        if (needRequest && requestParams) {
            loadData();
        }
    }, [loadData, needRequest, requestParams]);

    return (
        <div className={cx('dynamics-chart')}>
            {searchIsInProcess && (
                <div className={cx('dynamics-chart__spinner')}>
                    <Spinner size="s" />
                    <div className={cx('dynamics-chart__spinner-text')}>
                        {i18nBlock.dynamicsDashSpinnerDashText()}
                    </div>
                </div>
            )}

            {searchIsFailed && (
                <div className={cx('dynamics-chart__error')}>
                    <div className={cx('dynamics-chart__error-text')}>
                        {i18nBlock.dynamicsDashUnknownDashErrorDashText()}
                    </div>
                    <div className={cx('dynamics-chart__retry-button')}>
                        <Button onClick={onRetryLoad}>
                            {i18nBlock.dynamicsDashRetryDashButtonDashText()}
                        </Button>
                    </div>
                </div>
            )}

            {searchIsSuccessful && requestParams && prices && (
                <ChartSearchSuccess
                    requestParams={requestParams}
                    prices={prices}
                    priceComparator={priceComparator}
                    mode={mode}
                    selectedDate={selectedDate}
                    onDaysScroll={onDaysScroll}
                    onDaySelect={onDaySelect}
                    onLinkClick={onLinkClick}
                    addData={addData}
                    view={view}
                    standalone={standalone}
                />
            )}
        </div>
    );
};

export default React.memo(ChartDynamic);
