import React, {useMemo} from 'react';
import {useDispatch} from 'react-redux';

import {
    IDynamicsDay,
    EDynamicsDaysMode,
    EDynamicsDaysShown,
} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import TValidDataForRequestDynamic from 'types/avia/dynamic/TValidDataForRequestDynamic';
import {TInlineSearchRequest} from 'projects/avia/components/Dynamics/types/TInlineSearchRequest';

import {TDynamicsDaysInfo} from 'reducers/avia/aviaPriceIndex/utils/convertPriceIndexDataToDynamicsData';
import {inlineSearchRequest} from 'reducers/avia/aviaPriceIndex/dynamics/inlineSearchRequest';

import {PriceComparator} from 'utilities/currency/compare';
import getDatesDelta from 'projects/avia/lib/dynamic/getDatesDelta';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {dynamicsPriceDays} from 'projects/avia/components/Dynamics/utilities/dynamicsPriceDays';
import {filterValuesToHash} from 'projects/avia/lib/search/filters/converters';
import {getAppliedAviaFilters} from 'projects/avia/lib/search/filters/getAppliedFilters';

import DynamicsDays, {
    TAddData,
} from '../../components/DynamicsDays/DynamicsDays';
import {TView} from 'projects/avia/components/Dynamics/Chart/DynamicsChart/ChartDynamic';

const shownDays = EDynamicsDaysShown.month * 3;

export interface IChartSearchSuccess {
    requestParams: TValidDataForRequestDynamic;
    prices: TDynamicsDaysInfo;
    priceComparator: PriceComparator;
    mode: EDynamicsDaysMode;
    selectedDate: Nullable<string>;
    onDaysScroll?: () => void;
    onDaySelect: (date: string) => void;
    onLinkClick: (
        date: IDynamicsDay,
        event: React.MouseEvent<HTMLLinkElement>,
    ) => void;
    addData: TAddData;

    view?: TView;
    standalone?: boolean;
}

const ChartSearchSuccess: React.FC<IChartSearchSuccess> = ({
    requestParams,
    prices,
    priceComparator,
    mode,
    selectedDate,
    onDaysScroll,
    onDaySelect,
    onLinkClick,
    addData,

    view,
    standalone,
}) => {
    const dispatch = useDispatch();
    const {searchForm, filters, interval} = requestParams;
    const delta = getDatesDelta(searchForm);

    const pricesKey = useMemo(
        () =>
            Object.keys(prices)
                .map(
                    key =>
                        `${key}${prices[key].price?.value}${prices[key].status}`,
                )
                .join(''),
        [prices],
    );

    // структура prices содержит незначимое для десктопного отображения поле progress,
    // которое при инлайн поиске часто меняет значение, хотя другие данные остаются без изменений
    // чтообы из-за него не происходило лишних апдейтов в зависимостях хука
    // смотрим не на исходный объект, а на его ключ
    const dynamicsDays = useMemo(
        () => dynamicsPriceDays(prices, delta, priceComparator),
        [pricesKey, delta, priceComparator],
    );
    const filtersHash = filterValuesToHash(getAppliedAviaFilters(filters));

    const inlineRequest: TInlineSearchRequest = useImmutableCallback(
        (forwardDate, backwardDate) => {
            dispatch(
                inlineSearchRequest(
                    forwardDate,
                    backwardDate,
                    searchForm,
                    filters,
                ),
            );
        },
    );

    const scrollDistance = {
        [EDynamicsDaysMode.MONTH]: standalone ? 14 : 7,
        [EDynamicsDaysMode.WEEK]: 1,
    };

    return (
        <DynamicsDays
            shownDays={shownDays}
            mode={mode}
            filtersHash={filtersHash}
            dynamicsDays={dynamicsDays.days}
            minPrice={dynamicsDays.minPrice}
            maxPrice={dynamicsDays.maxPrice}
            inlineSearchRequest={inlineRequest}
            forwardDate={selectedDate || searchForm.when}
            intervalWithData={interval}
            addData={addData}
            delta={delta}
            view={view}
            onDaysScroll={onDaysScroll}
            onDaySelect={onDaySelect}
            onLinkClick={onLinkClick}
            searchForm={searchForm}
            priceComparator={priceComparator}
            scrollDistanceFromProps={scrollDistance}
        />
    );
};

export default React.memo(ChartSearchSuccess);
