import React, {ReactNode, Component, RefObject} from 'react';
import moment from 'moment';

import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';
import {
    EDynamicsDaysMode,
    EDynamicsDaysShown,
    EDynamicsDayStatus,
    IDynamicsDay,
} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import {TInlineSearchRequest} from 'projects/avia/components/Dynamics/types/TInlineSearchRequest';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {ROBOT} from 'utilities/dateUtils/formats';
import IPrice from 'utilities/currency/PriceInterface';
import {PriceComparator} from 'utilities/currency/compare';
import {dynamicsDateFormat} from 'projects/avia/components/Dynamics/utilities/dynamicsDateFormat';

import * as i18nBlock from 'i18n/avia-AviaDynamics';

import Price from 'components/Price/Price';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';
import CarouselButton from 'components/Carousel/components/CarouselButton/CarouselButton';
import {DYNAMICS_DAY_SIZES} from './components/DynamicsDay/DynamicsDay';
import VirtualizedDays from './components/VirtualizedDays/VirtualizedDays';

import cx from './DynamicsDays.scss';

enum EDynamicsScrollDirection {
    LEFT = 'left',
    RIGHT = 'right',
}

export type TAddData = (startDate: string, endDate: string) => void;

export interface IDynamicsDaysProps {
    shownDays: number;
    mode: EDynamicsDaysMode;
    filtersHash: string;
    dynamicsDays: IDynamicsDay[];
    minPrice?: IPrice;
    maxPrice?: IPrice;
    inlineSearchRequest: TInlineSearchRequest;
    forwardDate: string;
    intervalWithData: {
        startDate: string;
        endDate: string;
    };
    onDaysScroll?: () => void;
    onDaySelect?: (date: string) => void;
    onLinkClick: (
        date: IDynamicsDay,
        event: React.MouseEvent<HTMLLinkElement>,
    ) => void;
    addData: TAddData;
    delta: Nullable<number>;
    searchForm: IAviaParams;
    view: 'modal' | 'yak';
    priceComparator: PriceComparator;
    scrollDistanceFromProps?: IDynamicsDaysScrollDistance;
}

export interface IDynamicsDaysScrollDistance {
    [EDynamicsDaysMode.WEEK]: number;
    [EDynamicsDaysMode.MONTH]: number;
}

const scrollDistance: IDynamicsDaysScrollDistance = {
    week: 1,
    month: 7,
};

const POPUP_DIRECTIONS = [EPopupDirection.TOP];

interface IDynamicsDaysState {
    currentItem: number;
    scrolledDays: number;
    hoveredDay: Nullable<{
        ref: Nullable<RefObject<HTMLElement>>;
        data: IDynamicsDay;
    }>;
    topOffsetFromWaterline?: number;
}

class DynamicsDays extends Component<IDynamicsDaysProps, IDynamicsDaysState> {
    static defaultProps = {
        view: 'modal',
    };

    state = this.getInitialState();

    private getInitialState(): IDynamicsDaysState {
        const currentItem = this.initCurrentItem();

        return {
            currentItem,
            scrolledDays: this.normalizeIndex(currentItem),
            hoveredDay: null,
            topOffsetFromWaterline: undefined,
        };
    }

    componentDidMount(): void {
        this.inlineSearch();
    }

    componentDidUpdate(
        prevProps: IDynamicsDaysProps,
        prevState: IDynamicsDaysState,
    ): void {
        // При изменении типа отображения возвращаем скролл к выбранному дню,
        // сбрасываем день для попапа
        if (this.props.mode !== prevProps.mode) {
            this.setState({
                scrolledDays: this.normalizeIndex(this.state.currentItem),
                hoveredDay: null,
            });
        }

        //
        if (this.state.scrolledDays === prevState.scrolledDays) {
            return;
        }

        // При скролле проверяем, запрошены ли цены для дней далее по направлению скролла
        const actualScrolledDays = this.state.scrolledDays;
        const prevScrolledDays = prevState.scrolledDays;

        const {mode, dynamicsDays, intervalWithData} = this.props;
        const shownDays = EDynamicsDaysShown[mode];

        const scrollDirection =
            actualScrolledDays > prevScrolledDays
                ? EDynamicsScrollDirection.RIGHT
                : EDynamicsScrollDirection.LEFT;

        // вычисляем крайний элемент скрола
        const lastItemInNextScrollIndex =
            actualScrolledDays +
            (scrollDirection === EDynamicsScrollDirection.RIGHT
                ? shownDays
                : -shownDays);

        // при скролле на границах динамики индекс крайнего дня будет выходит за границы
        // массива дней, в таком случае выходим из функции, т.к. данные до текущего дня или
        // больше года не показываем, а промахнуться не можем, т.к. дозагружаем данные по целым месяцам
        if (
            lastItemInNextScrollIndex <= 0 ||
            lastItemInNextScrollIndex >= dynamicsDays.length
        ) {
            return;
        }

        const lastDayInNextScroll = dynamicsDays[lastItemInNextScrollIndex];

        const lastDate = moment.utc(lastDayInNextScroll.dateForward.date);

        if (scrollDirection === EDynamicsScrollDirection.RIGHT) {
            const endDate = moment.utc(intervalWithData.endDate);

            if (lastDate.isAfter(endDate)) {
                this.props.addData(
                    intervalWithData.endDate,
                    lastDate.add(2, 'month').endOf('month').format(ROBOT),
                );
            }
        } else {
            const startDate = moment.utc(intervalWithData.startDate);

            if (lastDate.isBefore(startDate)) {
                this.props.addData(
                    lastDate
                        .subtract(2, 'month')
                        .startOf('month')
                        .format(ROBOT),
                    intervalWithData.startDate,
                );
            }
        }
    }

    private initCurrentItem(): number {
        const {dynamicsDays, forwardDate} = this.props;

        return (
            dynamicsDays.findIndex(
                day => day.dateForward.date === forwardDate,
            ) || 0
        );
    }

    private normalizeIndex(index: number): number {
        const {mode} = this.props;
        const halfMode = Math.floor(EDynamicsDaysShown[mode] / 2);

        return Math.max(0, index - halfMode);
    }

    private getPlaceholderStyles(
        size: {offset: number; width: number},
        scrollSpeed: number,
    ): {width: number; flexShrink: number} {
        return {
            width:
                (size.offset + size.width) *
                Math.max(0, this.state.scrolledDays - scrollSpeed),
            flexShrink: 0,
        };
    }

    private renderAdditional(): ReactNode {
        const {mode} = this.props;

        if (mode === EDynamicsDaysMode.WEEK) {
            return null;
        }

        return this.renderPopup();
    }

    private renderPopup(): ReactNode {
        const {hoveredDay} = this.state;

        if (!hoveredDay?.ref?.current) {
            return null;
        }

        return (
            <MessageBoxPopup
                direction={POPUP_DIRECTIONS}
                anchorRef={hoveredDay.ref}
                isVisible
            >
                <div className={cx('dynamics-days__popup-content')}>
                    {this.getPopupContent()}
                </div>
            </MessageBoxPopup>
        );
    }

    private getPopupContent(): ReactNode {
        const {hoveredDay} = this.state;

        if (!hoveredDay?.ref?.current) {
            return null;
        }

        if (!hoveredDay.data.price) {
            return (
                <div className={cx('dynamics-days__popup-findeable')}>
                    {i18nBlock.dynamicsDashPopupDashClickDashToDashFind()}
                </div>
            );
        }

        const {dateForward, dateBackward, price} = hoveredDay.data;

        const dateForwardMoment = moment.utc(dateForward.date);
        const dateBackwardMoment = dateBackward
            ? moment.utc(dateBackward.date)
            : null;

        return (
            <>
                <div className={cx('dynamics-days__popup-direction')}>
                    {dateBackwardMoment
                        ? i18nBlock.dynamicsDashPopupDashDirectionDashForwardDashBackward()
                        : i18nBlock.dynamicsDashPopupDashDirectionDashOnlyDashForward()}
                </div>
                <div className={cx('dynamics-days__popup-date')}>
                    {dynamicsDateFormat(dateForwardMoment, dateBackwardMoment)}
                </div>
                <div className={cx('dynamics-days__popup-price')}>
                    <Price {...price} isRoughly={price.roughly} />
                </div>
            </>
        );
    }

    private onDaySelect = (
        index: number,
        topOffsetFromWaterline?: number,
    ): void => {
        const {dynamicsDays, onDaySelect} = this.props;

        this.setState({
            currentItem: index,
        });
        this.tryUpdateWaterLine(topOffsetFromWaterline);

        if (onDaySelect) {
            onDaySelect(dynamicsDays[index].dateForward.date);
        }
    };

    private tryUpdateWaterLine = (topOffsetFromWaterline?: number): void => {
        if (topOffsetFromWaterline !== undefined) {
            this.setState({
                topOffsetFromWaterline,
            });
        }
    };

    private onDayHover = (
        colorColumn: RefObject<HTMLElement>,
        day: IDynamicsDay,
    ): void => {
        this.setState({
            hoveredDay: {
                ref: colorColumn,
                data: day,
            },
        });
    };

    private onHoverLeave = (): void => {
        this.setState({
            hoveredDay: null,
        });
    };

    private prepareRight(): number {
        const {scrolledDays} = this.state;
        const {mode} = this.props;
        const size = DYNAMICS_DAY_SIZES[mode];

        return (size.offset + size.width) * scrolledDays;
    }

    private scrollRight = (): void => {
        const {onDaysScroll} = this.props;

        this.setState(({scrolledDays}, {dynamicsDays, mode}) => ({
            scrolledDays: Math.min(
                dynamicsDays.length - EDynamicsDaysShown[mode],
                scrolledDays + this.getOffset(),
            ),
        }));

        onDaysScroll?.();
    };

    private scrollLeft = (): void => {
        const {onDaysScroll} = this.props;

        this.setState(({scrolledDays}) => ({
            scrolledDays: Math.max(scrolledDays - this.getOffset(), 0),
        }));

        onDaysScroll?.();
    };

    private getOffset(): number {
        const {mode, scrollDistanceFromProps} = this.props;

        if (scrollDistanceFromProps) {
            return scrollDistanceFromProps[mode];
        }

        return scrollDistance[mode];
    }

    private inlineSearch(): void {
        const {currentItem} = this.state;
        const {dynamicsDays, inlineSearchRequest} = this.props;

        const daysToSearch = 7;

        const start = currentItem < 3 ? 0 : currentItem - 3;
        const end = start + daysToSearch;

        for (let i = start; i < end; i++) {
            const day = dynamicsDays[i];

            if (day.status === EDynamicsDayStatus.SHOULD_SEARCH) {
                inlineSearchRequest(
                    day.dateForward.date,
                    day.dateBackward ? day.dateBackward.date : null,
                );
            }
        }
    }

    render(): ReactNode {
        const {
            dynamicsDays,
            mode,
            delta,
            minPrice,
            maxPrice,
            searchForm,
            filtersHash,
            priceComparator,
            inlineSearchRequest,
            onLinkClick,
        } = this.props;
        const {topOffsetFromWaterline, scrolledDays, currentItem} = this.state;
        const right = this.prepareRight();
        const style = {
            right: `${right}px`,
        };

        const size = DYNAMICS_DAY_SIZES[mode];
        const partSize = EDynamicsDaysShown[mode];
        const scrollSpeed = scrollDistance[mode];

        const placeholderStyles = this.getPlaceholderStyles(size, scrollSpeed);
        const hasPrevious = scrolledDays > 0;
        const hasNext =
            scrolledDays + EDynamicsDaysShown[mode] < dynamicsDays.length - 1;

        const styleForWaterline = {
            top: `${topOffsetFromWaterline}px`,
            left: style.right,
        };

        return (
            <div
                className={cx(
                    'dynamics-days',
                    `_mode-${this.props.mode}`,
                    `_view_${this.props.view}`,
                )}
                onMouseLeave={this.onHoverLeave}
            >
                <div className={cx('dynamics-days__arrow', '_left')}>
                    {hasPrevious && (
                        <CarouselButton
                            onClick={this.scrollLeft}
                            type="prev"
                            arrowsPosition="center"
                            arrowsType="light"
                        />
                    )}
                </div>
                <div className={cx('dynamics-days__overflow-wrapper')}>
                    <div
                        className={cx('dynamics-days__scroll-wrapper')}
                        style={style}
                    >
                        <div className={cx('dynamics-days__container')}>
                            {topOffsetFromWaterline !== undefined && (
                                <div
                                    className={cx('dynamics-days__waterline')}
                                    style={styleForWaterline}
                                />
                            )}
                            <div style={placeholderStyles} />
                            <VirtualizedDays
                                start={Math.max(0, scrolledDays - scrollSpeed)}
                                end={scrolledDays + partSize + scrollSpeed}
                                mode={mode}
                                delta={delta}
                                minPrice={minPrice}
                                maxPrice={maxPrice}
                                searchForm={searchForm}
                                filtersHash={filtersHash}
                                dynamicsDays={dynamicsDays}
                                currentItem={currentItem}
                                priceComparator={priceComparator}
                                inlineSearchRequest={inlineSearchRequest}
                                onLinkClick={onLinkClick}
                                onDaySelect={this.onDaySelect}
                                onDayHover={this.onDayHover}
                            />
                        </div>
                        <div className={cx('dynamics-days__additional')}>
                            {this.renderAdditional()}
                        </div>
                    </div>
                </div>
                <div className={cx('dynamics-days__arrow', '_right')}>
                    {hasNext && (
                        <CarouselButton
                            onClick={this.scrollRight}
                            type="next"
                            arrowsPosition="center"
                            arrowsType="light"
                        />
                    )}
                </div>
            </div>
        );
    }
}

export default DynamicsDays;
