import React, {ReactNode, PureComponent, createRef, RefObject} from 'react';
import moment from 'moment';

import {IAviaPriceIndexPrice} from 'server/api/AviaPriceIndexApi/types/IAviaPriceIndexPrice';
import {
    EDynamicsDaysMode,
    EDynamicsDayStatus,
    IDynamicsDay,
} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import {TInlineSearchRequest} from 'projects/avia/components/Dynamics/types/TInlineSearchRequest';

import {CHAR_DASH} from 'utilities/strings/charCodes';
import {DATE, HUMAN_WEEKDAY_SHORT, MONTH} from 'utilities/dateUtils/formats';
import {dynamicsDateFormat} from 'projects/avia/components/Dynamics/utilities/dynamicsDateFormat';

import * as i18nBlock from 'i18n/avia-AviaDynamics';

import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import Button from 'components/Button/Button';
import Spinner from 'components/Spinner/Spinner';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import CancelIcon from 'icons/16/Cancel';

import cx from './DynamicsDay.scss';

export const DYNAMICS_DAY_SIZES = {
    week: {
        offset: 4,
        width: 152,
    },
    month: {
        offset: 2,
        width: 32,
    },
};

interface IDynamicsDayProps {
    height: number;
    mode: EDynamicsDaysMode;
    onSearch: TInlineSearchRequest;
    isCurrent: boolean;
    index: number;
    onSelect: (index: number, topOffset?: number) => void;
    onLinkClick: (
        date: IDynamicsDay,
        event: React.MouseEvent<HTMLLinkElement>,
    ) => void;
    onHover: (
        colorColumnRef: RefObject<HTMLElement>,
        day: IDynamicsDay,
    ) => void;
    isInInterval: boolean;
    showMonth: boolean;
    dayUrl: string;
    day: IDynamicsDay;
    isCheapest?: boolean;
    organicAviaDynamicMinPrice?: boolean;
}

class DynamicsDay extends PureComponent<IDynamicsDayProps> {
    private colorColumn: Nullable<HTMLElement> = null;
    private graphColumn: Nullable<HTMLElement> = null;

    private columnRef = createRef<HTMLDivElement>();

    componentDidMount(): void {
        this.colorColumn = this.columnRef.current;

        if (this.props.isCurrent) {
            const topOffset = this.calculateTopOffset();

            this.props.onSelect(this.props.index, topOffset);
        }
    }

    private renderIcon(): ReactNode {
        const {mode} = this.props;
        let icon;

        if (this.statusIsNoDataOrError()) {
            icon = <CancelIcon />;
        }

        if (this.props.day.status === EDynamicsDayStatus.SEARCHING) {
            icon = <Spinner size="xxs" />;
        }

        if (icon) {
            return (
                <div className={cx('icon')}>
                    {icon}
                    {mode === EDynamicsDaysMode.WEEK && (
                        <div>{i18nBlock.dynamicsLandingsSearchingPrices()}</div>
                    )}
                </div>
            );
        }

        return null;
    }

    private renderWeekDateInfo(): ReactNode {
        const {
            day: {dateForward, dateBackward},
        } = this.props;

        const dateForwardMoment = moment.utc(dateForward.date);

        let dateBackwardMoment = null;

        if (dateBackward && dateBackward.date) {
            dateBackwardMoment = moment.utc(dateBackward.date);
        }

        const monthInfo = dynamicsDateFormat(
            dateForwardMoment,
            dateBackwardMoment,
        );

        return (
            <Text className={cx('extendedDateInfo')} tag="div" size="m">
                <div>{monthInfo}</div>
                <div className={cx('weekdayName')}>
                    <Text color={dateForward.isHoliday ? 'alert' : undefined}>
                        {dateForwardMoment.format(HUMAN_WEEKDAY_SHORT)}
                    </Text>
                    {dateBackward && dateBackwardMoment && (
                        <>
                            <div>{CHAR_DASH}</div>
                            <Text
                                color={
                                    dateBackward.isHoliday ? 'alert' : undefined
                                }
                            >
                                {dateBackwardMoment.format(HUMAN_WEEKDAY_SHORT)}
                            </Text>
                        </>
                    )}
                </div>
            </Text>
        );
    }

    private renderMonthModeInfo(): ReactNode {
        const {
            day: {dateForward, dateBackward},
            showMonth,
            isCurrent,
            isInInterval,
        } = this.props;
        const dateForwardMoment = moment.utc(dateForward.date);
        const dateBackwardMoment = dateBackward?.date
            ? moment.utc(dateBackward.date)
            : null;

        return (
            <>
                <Text
                    size="s"
                    color={dateForward.isHoliday ? 'alert' : 'current'}
                    weight={isCurrent || isInInterval ? 'bold' : undefined}
                >
                    {dateForwardMoment.format(DATE)}
                </Text>
                <div className={cx('weekdayName')}>
                    <Text color={dateForward.isHoliday ? 'alert' : 'current'}>
                        {dateForwardMoment.format(HUMAN_WEEKDAY_SHORT)}
                    </Text>
                    {dateBackward && dateBackwardMoment && (
                        <>
                            <div>{CHAR_DASH}</div>
                            <Text
                                color={
                                    dateBackward.isHoliday ? 'alert' : 'current'
                                }
                            >
                                {dateBackwardMoment.format(HUMAN_WEEKDAY_SHORT)}
                            </Text>
                        </>
                    )}
                </div>
                {showMonth && (
                    <div className={cx('monthName')}>
                        {dateForwardMoment.format(MONTH)}
                    </div>
                )}
            </>
        );
    }

    private onClick = (): void => {
        const {
            day: {status, dateForward, dateBackward},
        } = this.props;

        if (status === EDynamicsDayStatus.SHOULD_SEARCH) {
            this.props.onSearch(
                dateForward.date,
                dateBackward ? dateBackward.date : null,
            );
        }

        if (status === EDynamicsDayStatus.HAS_DATA) {
            const topOffset = this.calculateTopOffset();

            this.props.onSelect(this.props.index, topOffset);
        }
    };

    private calculateTopOffset = (): number | undefined => {
        if (!this.graphColumn || !this.colorColumn) {
            return undefined;
        }

        return (
            parseFloat(window.getComputedStyle(this.graphColumn, null).height) -
            parseFloat(window.getComputedStyle(this.colorColumn, null).height)
        );
    };

    private onLinkClick = (e: React.MouseEvent<HTMLLinkElement>): void => {
        const {onLinkClick, day} = this.props;

        onLinkClick(day, e);
    };

    private onHover = (): void => {
        const {onHover} = this.props;
        const {
            day: {status, dateForward, dateBackward, price, progress},
        } = this.props;

        if (this.statusIsNoDataOrError()) {
            return;
        }

        onHover(this.columnRef, {
            dateForward,
            dateBackward,
            progress,
            status,
            price,
        });
    };

    /**
     * TODO
     * Desktop версия не различает EDynamicsDayStatus.NO_DATA и EDynamicsDayStatus.WAS_ERROR
     * Стоит сделать по аналогии с тачом - https://st.yandex-team.ru/TRAVELFRONT-2162
     */
    private statusIsNoDataOrError = (): boolean => {
        const {
            day: {status},
        } = this.props;

        return (
            status === EDynamicsDayStatus.NO_DATA ||
            status === EDynamicsDayStatus.WAS_ERROR
        );
    };

    private renderNoPrice(): ReactNode {
        return (
            <Text className={cx('noPriceInfo')} size="s" weight="bold">
                {i18nBlock.dynamicsDashDayDashNoDashFlights()}
            </Text>
        );
    }

    private renderLinkButton = (children: ReactNode): ReactNode => {
        return (
            <ButtonLink url={this.props.dayUrl} onClick={this.onLinkClick}>
                {children}
            </ButtonLink>
        );
    };

    private renderFindButton = (children: ReactNode): ReactNode => {
        return <Button onClick={this.onClick}>{children}</Button>;
    };

    private renderDayButton(price: Nullable<IAviaPriceIndexPrice>): ReactNode {
        const {
            day: {status},
        } = this.props;
        const child = price ? (
            <Price {...price} isFrom isRoughly={price.roughly} />
        ) : (
            i18nBlock.dynamicsDashDayDashButtonDashFind()
        );
        const isSearchable =
            status === EDynamicsDayStatus.SEARCHING ||
            status === EDynamicsDayStatus.SHOULD_SEARCH;

        return (
            <div>
                {isSearchable
                    ? this.renderFindButton(child)
                    : this.renderLinkButton(child)}
            </div>
        );
    }

    private renderPriceInfo(): ReactNode {
        const {
            day: {price},
        } = this.props;

        return this.statusIsNoDataOrError()
            ? this.renderNoPrice()
            : this.renderDayButton(price);
    }

    private renderWeekModeInfo(): ReactNode {
        return (
            <>
                {this.renderPriceInfo()}
                {this.renderWeekDateInfo()}
            </>
        );
    }

    private renderInfo(): ReactNode {
        return this.props.mode === EDynamicsDaysMode.WEEK
            ? this.renderWeekModeInfo()
            : this.renderMonthModeInfo();
    }

    private getGraphState():
        | 'currentShouldSearch'
        | 'shouldSearch'
        | 'inSearchProgress'
        | 'current'
        | 'cheapest'
        | 'empty'
        | 'default' {
        const {
            isCurrent,
            isCheapest,
            day: {status, price},
            organicAviaDynamicMinPrice,
        } = this.props;

        const dayStatus = price ? EDynamicsDayStatus.HAS_DATA : status;

        switch (true) {
            case isCurrent && dayStatus === EDynamicsDayStatus.SHOULD_SEARCH:
                return 'currentShouldSearch';

            case dayStatus === EDynamicsDayStatus.SHOULD_SEARCH:
                return 'shouldSearch';

            case dayStatus === EDynamicsDayStatus.SEARCHING:
                return 'inSearchProgress';

            case isCurrent:
                return 'current';

            case isCheapest && organicAviaDynamicMinPrice:
                return 'cheapest';

            case dayStatus === EDynamicsDayStatus.HAS_DATA:
                return 'default';

            case dayStatus === EDynamicsDayStatus.NO_DATA:
            default:
                return 'empty';
        }
    }

    render(): ReactNode {
        const {mode, isCurrent} = this.props;

        return (
            <div className={cx('root', `root_mode_${mode}`)}>
                <div
                    className={cx('graph')}
                    onClick={this.onClick}
                    ref={(el): void => {
                        this.graphColumn = el;
                    }}
                >
                    <div
                        onMouseOver={this.onHover}
                        className={cx(
                            'graphColoredPart',
                            `graphColoredPart_state_${this.getGraphState()}`,
                        )}
                        style={{
                            height: `${this.props.height}%`,
                        }}
                        ref={this.columnRef}
                    >
                        {this.renderIcon()}
                    </div>
                </div>
                <div
                    className={cx('info', {
                        info_view_card:
                            mode === EDynamicsDaysMode.WEEK && isCurrent,
                    })}
                >
                    {this.renderInfo()}
                </div>
            </div>
        );
    }
}

export default DynamicsDay;
