import React, {RefObject, useMemo} from 'react';
import moment from 'moment';
import {parse} from 'query-string';

import {
    EDynamicsDaysMode,
    IDynamicsDay,
} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import {TInlineSearchRequest} from 'projects/avia/components/Dynamics/types/TInlineSearchRequest';
import {
    EOneWay,
    IAviaParams,
} from 'server/services/AviaSearchService/types/IAviaParams';
import {EAviaLinkSource} from 'types/avia/url/EAviaLinkSource';

import IPrice from 'utilities/currency/PriceInterface';
import {PriceComparator} from 'utilities/currency/compare';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {aviaURLs} from 'projects/avia/lib/urls';
import {isValidDate} from 'utilities/dateUtils';
import {getDynamicsDayHeight} from 'projects/avia/components/Dynamics/Chart/DynamicsChart/components/DynamicsDays/components/VirtualizedDays/utilities/getDynamicsDayHeight';

import DynamicsDay from 'projects/avia/components/Dynamics/Chart/DynamicsChart/components/DynamicsDays/components/DynamicsDay/DynamicsDay';

interface IVirtualizedDaysProps {
    start: number;
    end: number;
    currentItem: number;
    mode: EDynamicsDaysMode;
    filtersHash: string;
    dynamicsDays: IDynamicsDay[];
    minPrice?: IPrice;
    maxPrice?: IPrice;
    delta: Nullable<number>;
    searchForm: IAviaParams;
    priceComparator: PriceComparator;
    inlineSearchRequest: TInlineSearchRequest;
    onDaySelect: (index: number, topOffset?: number) => void;
    onLinkClick: (
        date: IDynamicsDay,
        event: React.MouseEvent<HTMLLinkElement>,
    ) => void;
    onDayHover: (
        colorColumnRef: RefObject<HTMLElement>,
        day: IDynamicsDay,
    ) => void;
}

const VirtualizedDays: React.FC<IVirtualizedDaysProps> = props => {
    const {
        start,
        end,
        mode,
        delta,
        minPrice,
        maxPrice,
        searchForm,
        filtersHash,
        dynamicsDays,
        currentItem,
        priceComparator,
        inlineSearchRequest,
        onLinkClick,
        onDaySelect,
        onDayHover,
    } = props;

    const {organicAviaDynamicMinPrice} = useExperiments();

    const days = useMemo(() => {
        const parsedHash = parse(filtersHash);

        return dynamicsDays.map((day, index) => {
            const isInInterval =
                delta === null
                    ? false
                    : index >= currentItem && currentItem + delta >= index;

            let showMonth = false;

            if (mode === EDynamicsDaysMode.MONTH) {
                const date = moment.utc(day.dateForward.date).date();

                // TODO: 26 is a magic number, move it in file constants
                showMonth = date === 1 || (index === 0 && date < 26);
            }

            const dayUrl = aviaURLs.getSearchResultsUrl(
                {
                    ...searchForm,
                    when: day.dateForward.date,
                    return_date: day.dateBackward?.date,
                    linkSource: EAviaLinkSource.DYNAMIC,
                    oneway:
                        day.dateBackward?.date &&
                        isValidDate(day.dateBackward.date)
                            ? EOneWay.ROUND_TRIP
                            : EOneWay.ONE_WAY,
                },
                parsedHash,
            );

            return (
                <DynamicsDay
                    key={day.dateForward.date}
                    index={index}
                    mode={mode}
                    height={getDynamicsDayHeight({
                        price: day.price,
                        minPrice,
                        maxPrice,
                        priceComparator,
                    })}
                    onSearch={inlineSearchRequest}
                    onSelect={onDaySelect}
                    onLinkClick={onLinkClick}
                    onHover={onDayHover}
                    isCurrent={currentItem === index}
                    isInInterval={isInInterval}
                    showMonth={showMonth}
                    dayUrl={dayUrl}
                    day={day}
                    isCheapest={priceComparator.isEqualPrice(
                        day.price || undefined,
                        minPrice,
                    )}
                    organicAviaDynamicMinPrice={organicAviaDynamicMinPrice}
                />
            );
        });
    }, [
        currentItem,
        delta,
        dynamicsDays,
        filtersHash,
        inlineSearchRequest,
        maxPrice,
        minPrice,
        mode,
        onDayHover,
        onDaySelect,
        onLinkClick,
        priceComparator,
        searchForm,
        organicAviaDynamicMinPrice,
    ]);

    return <>{days.slice(start, end)}</>;
};

export default React.memo(VirtualizedDays);
