import React, {Component, ReactNode} from 'react';
import {connect, DispatchProp} from 'react-redux';

import {
    EDynamicsDaysMode,
    IDynamicsDay,
} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import {IWithClassName} from 'types/withClassName';
import EAviaDynamicActionLogActionName from 'server/loggers/avia/AviaActionLog/types/EAviaDynamicActionLogActionName';
import EAviaDynamicTypeView from 'types/avia/priceIndex/EAviaDynamicTypeView';

import * as actions from 'reducers/avia/aviaPriceIndex/dynamics/actions';
import {StoreInterface} from 'reducers/storeTypes';
import {loggerActions} from 'reducers/avia/aviaLogging/actions';

import {priceComparatorSelector} from 'selectors/common/currenciesSelector';
import {aviaContextSelector} from 'selectors/avia/context/aviaContextSelector';
import {aviaDynamicsPricesSearchFormSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesSearchFormSelector';
import {aviaDynamicsPricesFinalPriceSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesFinalPriceSelector';
import {aviaDynamicsPricesTypeViewSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesSelectors';

import getValueFromEnum from 'utilities/getValueFromEnum';
import {dynamicsPriceDays} from 'projects/avia/components/Dynamics/utilities/dynamicsPriceDays';
import getDatesDelta from 'projects/avia/lib/dynamic/getDatesDelta';
import isCurrentDynamicDay from '../../utilities/isCurrentDynamicDay';

import * as i18nBlock from 'i18n/avia-AviaDynamics';

import Box from 'components/Box/Box';
import Heading from 'components/Heading/Heading';
import RadioButton, {
    TOnChangeRadioButton,
} from 'components/RadioButton/RadioButton';
import RadioButtonSlide from 'components/RadioButtonSlide/RadioButtonSlide';
import AviaDynamicsFooter from 'projects/avia/components/Dynamics/DynamicsFooter/DynamicsFooter';
import StandaloneFilters from 'projects/avia/components/Dynamics/StandaloneFilters/StandaloneFilters';
import {DynamicsFiltersConnected} from 'projects/avia/components/Dynamics/Chart/DynamicsFilters/DynamicsFiltersConnected';
import DynamicsDeltaButtonConnected from 'projects/avia/components/Dynamics/DynamicsDeltaButton/DynamicsDeltaButtonConnected';
import DynamicTable from 'projects/avia/components/Dynamics/Table/DynamicTable';
import ChartDynamic from 'projects/avia/components/Dynamics/Chart/DynamicsChart/ChartDynamic';

import cx from './DynamicsCommonContainer.scss';

const mapStateToProps = (state: StoreInterface) => {
    const prices = aviaDynamicsPricesFinalPriceSelector(state);
    const typeView = aviaDynamicsPricesTypeViewSelector(state);
    const searchForm = aviaDynamicsPricesSearchFormSelector(state);
    const delta = getDatesDelta(searchForm);
    const priceComparator = priceComparatorSelector(state);
    const aviaContext = aviaContextSelector(state);
    const dynamicsDays = prices
        ? dynamicsPriceDays(prices, delta, priceComparator).days
        : [];

    return {
        dynamicsDays,
        searchForm,
        aviaContext,
        typeView,
    };
};

interface IOwnCommonContainerProps extends IWithClassName {
    view: 'modal' | 'yak';
    standalone?: boolean;
    onClose?: () => void;
    doNotPreventLinkClick?: boolean;
    newDesktopDynamic?: boolean; // Эксперимент с новой динамикой в десктопе
    mode?: EDynamicsDaysMode;

    onSearch?: () => void;
    onFilterChange?: () => void;
    onScroll?: () => void;
    onModeChange?: (mode: EDynamicsDaysMode) => void;
}

type TDynamicsCommonContainerProps = ReturnType<typeof mapStateToProps> &
    IOwnCommonContainerProps &
    DispatchProp;

interface IDynamicsPriceSelectorState {
    mode: EDynamicsDaysMode;
    selectedDate: Nullable<string>;
}

class DynamicsCommonContainer extends Component<
    TDynamicsCommonContainerProps,
    IDynamicsPriceSelectorState
> {
    static defaultProps: Pick<IOwnCommonContainerProps, 'view'> = {
        view: 'modal',
    };

    constructor(props: TDynamicsCommonContainerProps) {
        super(props);

        this.state = {
            mode: this.props.mode || EDynamicsDaysMode.WEEK,
            selectedDate: this.props.searchForm.when,
        };
    }

    private handleSearch = (): void => {
        const {onSearch} = this.props;

        onSearch?.();
    };

    private handleFilterChange = (): void => {
        const {onFilterChange} = this.props;

        onFilterChange?.();
    };

    private handleScroll = (): void => {
        const {onScroll} = this.props;

        onScroll?.();
    };

    private handleModeChange = (value: string): void => {
        const {onModeChange} = this.props;
        const mode = getValueFromEnum(value, EDynamicsDaysMode);

        if (!mode) {
            return;
        }

        this.setState({mode});

        const {dispatch} = this.props;

        if (mode === EDynamicsDaysMode.MONTH) {
            dispatch(
                loggerActions.logDynamicAction({
                    name: EAviaDynamicActionLogActionName.DYNAMIC_MONTH_SWITCH,
                }),
            );
        }

        if (mode === EDynamicsDaysMode.WEEK) {
            dispatch(
                loggerActions.logDynamicAction({
                    name: EAviaDynamicActionLogActionName.DYNAMIC_WEEK_SWITCH,
                }),
            );
        }

        onModeChange?.(mode);
    };

    private handleDayChange = (date: string): void => {
        this.setState({
            selectedDate: date,
        });
    };

    private onLinkClick = (
        date: IDynamicsDay,
        event: React.MouseEvent<HTMLLinkElement>,
    ): void => {
        const {aviaContext, doNotPreventLinkClick, onClose} = this.props;

        if (isCurrentDynamicDay(aviaContext, date) && !doNotPreventLinkClick) {
            onClose?.();
            event.preventDefault();
        } else {
            this.handleSearch();
        }
    };

    private onChangeTypeView: TOnChangeRadioButton = event => {
        const typeView = getValueFromEnum(
            event.currentTarget.value,
            EAviaDynamicTypeView,
        );

        if (typeView) {
            this.props.dispatch(actions.setTypeView(typeView));
        }
    };

    render(): ReactNode {
        const {mode, selectedDate} = this.state;
        const {
            searchForm,
            dynamicsDays,
            view,
            className,
            standalone = false,
            newDesktopDynamic,
            typeView: typeViewFromReduxState,
        } = this.props;
        const selectedDay = dynamicsDays.find(
            ({dateForward}) => dateForward.date === selectedDate,
        );
        const isOneWay = !searchForm.return_date;
        const typeView: EAviaDynamicTypeView = newDesktopDynamic
            ? typeViewFromReduxState
            : EAviaDynamicTypeView.CHART;

        return (
            <div
                className={cx(
                    className,
                    'root',
                    `root_view_${view}`,
                    newDesktopDynamic && 'root_newDesktopDynamic',
                    `root_typeView_${
                        typeView === EAviaDynamicTypeView.TABLE
                            ? 'table'
                            : 'chart'
                    }`,
                    isOneWay && 'root_isOneWay',
                )}
            >
                <div className={cx('title')}>
                    <Heading level={1}>
                        {i18nBlock.dynamicsDashContainerDashTitle()}
                    </Heading>

                    {newDesktopDynamic && (
                        <RadioButton
                            onChange={this.onChangeTypeView}
                            value={typeView}
                            className={cx('radioView')}
                        >
                            <RadioButton.Radio
                                value={EAviaDynamicTypeView.TABLE}
                            >
                                {i18nBlock.dynamicDashPriceDashTable()}
                            </RadioButton.Radio>
                            <RadioButton.Radio
                                value={EAviaDynamicTypeView.CHART}
                            >
                                {i18nBlock.dynamicDashPriceDashChart()}
                            </RadioButton.Radio>
                        </RadioButton>
                    )}
                </div>

                <div className={cx('controls')}>
                    {!newDesktopDynamic && (
                        <Box inline between="5" below="5">
                            <RadioButtonSlide
                                className={cx('modeToggler')}
                                value={mode}
                                name="dynamics-mode"
                                onChange={this.handleModeChange}
                                size="m"
                            >
                                <RadioButtonSlide.Option
                                    value={EDynamicsDaysMode.WEEK}
                                >
                                    {i18nBlock.dynamicsDashModeDashWeek()}
                                </RadioButtonSlide.Option>

                                <RadioButtonSlide.Option
                                    value={EDynamicsDaysMode.MONTH}
                                >
                                    {i18nBlock.dynamicsDashModeDashMonth()}
                                </RadioButtonSlide.Option>
                            </RadioButtonSlide>

                            <DynamicsDeltaButtonConnected />
                        </Box>
                    )}

                    <div className={cx('filters')}>
                        <div className={cx('filtersContainer')}>
                            {standalone ? (
                                <StandaloneFilters
                                    onChange={this.handleFilterChange}
                                />
                            ) : (
                                <DynamicsFiltersConnected />
                            )}
                        </div>
                        {newDesktopDynamic &&
                            typeView === EAviaDynamicTypeView.CHART && (
                                <DynamicsDeltaButtonConnected
                                    className={cx('deltaButtons')}
                                />
                            )}
                    </div>
                </div>

                {!newDesktopDynamic && <hr className={cx('hr')} />}

                {!newDesktopDynamic ||
                typeView === EAviaDynamicTypeView.CHART ? (
                    <div className={cx('chart')}>
                        <ChartDynamic
                            view={this.props.view}
                            mode={mode}
                            selectedDate={selectedDate}
                            onDaysScroll={this.handleScroll}
                            onDaySelect={this.handleDayChange}
                            onLinkClick={this.onLinkClick}
                            standalone={standalone}
                        />
                    </div>
                ) : (
                    <>
                        <div className={cx('fakeBlock')} />

                        <DynamicTable
                            onLinkClick={this.handleSearch}
                            className={cx('table')}
                        />
                    </>
                )}

                {mode === EDynamicsDaysMode.MONTH && (
                    <div className={cx('footer')}>
                        {selectedDay && (
                            <AviaDynamicsFooter
                                day={selectedDay}
                                searchForm={searchForm}
                                onLinkClick={this.onLinkClick}
                                useFiltersFromUrl={standalone}
                            />
                        )}
                    </div>
                )}
            </div>
        );
    }
}

export default connect(mapStateToProps)(DynamicsCommonContainer);
