import {useCallback} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import EAviaDynamicActionLogActionName from 'server/loggers/avia/AviaActionLog/types/EAviaDynamicActionLogActionName';

import {
    EAviaSearchTransferFilter,
    IAviaSearchResultsFilters,
} from 'reducers/avia/search/results/filters/reducer';
import {loggerActions} from 'reducers/avia/aviaLogging/actions';

import {getAviaSearchFilters} from 'selectors/avia/search/aviaSearchResultsSelector';
import {aviaContextSelector} from 'selectors/avia/context/aviaContextSelector';

import {Flag, isEnabled} from 'utilities/flags/flags';
import {useAviaBaggageFilter} from 'projects/avia/lib/hooks/filters/baggage';
import {useAviaNoTransfersFilter} from 'projects/avia/lib/hooks/filters/noTransfers';
import {useAviaTransfersFilter} from 'projects/avia/lib/hooks/filters/transfers';
import {isValidDate} from 'utilities/dateUtils';

import * as i18nBlock from 'i18n/avia-search';

import Box from 'components/Box/Box';
import {AviaTransferFilterPopup} from 'projects/avia/components/AviaSearchResultsTransferFilter/AviaTransferFilterPopup/AviaTransferFilterPopup';
import {AviaSearchResultsNoTransferFilter} from 'projects/avia/components/AviaSearchResultsNoTransferFilter/AviaSearchResultsNoTransferFilter';
import {AviaSearchResultsBaggageFilter} from 'projects/avia/components/AviaSearchResultsBaggageFilter/AviaSearchResultsBaggageFilter';
import DropdownFilter from 'components/DropdownFilter/DropdownFilter';
import AviaSearchResultsTimeFilter from 'projects/avia/components/AviaSearchResultsTimeFilter/AviaSearchResultsTimeFilter';

function DynamicsNoTransfersFilter() {
    const {
        filter: {value},
        handle,
    } = useAviaNoTransfersFilter();
    const dispatch = useDispatch();

    const checked = isEnabled(
        value.value,
        EAviaSearchTransferFilter.NO_TRANSFERS,
    );

    const onChange = useCallback(() => {
        handle();
        dispatch(
            loggerActions.logDynamicAction({
                name: checked
                    ? EAviaDynamicActionLogActionName.DYNAMIC_NO_TRANSFERS_OFF
                    : EAviaDynamicActionLogActionName.DYNAMIC_NO_TRANSFERS_ON,
            }),
        );
    }, [checked, dispatch, handle]);

    return (
        <AviaSearchResultsNoTransferFilter
            checked={checked}
            onChange={onChange}
        />
    );
}

function DynamicsBaggageFilter() {
    const {
        filter: {
            value: {enabled},
        },
        handle: handleBaggageChange,
    } = useAviaBaggageFilter();
    const dispatch = useDispatch();

    const onChange = useCallback(() => {
        handleBaggageChange();
        dispatch(
            loggerActions.logDynamicAction({
                name: enabled
                    ? EAviaDynamicActionLogActionName.DYNAMIC_BAGGAGE_OFF
                    : EAviaDynamicActionLogActionName.DYNAMIC_BAGGAGE_ON,
            }),
        );
    }, [dispatch, enabled, handleBaggageChange]);

    return (
        <AviaSearchResultsBaggageFilter onChange={onChange} checked={enabled} />
    );
}

function DynamicsTransferFilter() {
    const label = i18nBlock.filterDotTransfersDashPlaceholder();
    const dispatch = useDispatch();
    const {
        filter: {
            value: {value},
        },
        handle,
    } = useAviaTransfersFilter();
    const active = value !== 0;

    const onClickButton = useCallback(() => {
        dispatch(
            loggerActions.logDynamicAction({
                name: EAviaDynamicActionLogActionName.DYNAMIC_TRANSFER_FILTER_CLICK,
            }),
        );
    }, [dispatch]);

    const onChangeFilter = useCallback(
        (newValue: Flag<EAviaSearchTransferFilter>) => {
            handle(newValue);
            dispatch(
                loggerActions.logDynamicAction({
                    name: EAviaDynamicActionLogActionName.DYNAMIC_TRANSFER_FILTER,
                }),
            );
        },
        [dispatch, handle],
    );

    return (
        <DropdownFilter
            checked={active}
            label={label}
            onClickButton={onClickButton}
        >
            <AviaTransferFilterPopup
                hideRange
                hideTransferAirports
                transfersValue={value}
                onChangeTransfers={onChangeFilter}
            />
        </DropdownFilter>
    );
}

export function DynamicsFiltersConnected() {
    const dispatch = useDispatch();
    const context = useSelector(aviaContextSelector);
    const isRoundtrip = Boolean(
        context.return_date && isValidDate(context.return_date),
    );
    const filters: IAviaSearchResultsFilters =
        useSelector(getAviaSearchFilters);

    const onClickButton = useCallback(() => {
        dispatch(
            loggerActions.logDynamicAction({
                name: EAviaDynamicActionLogActionName.DYNAMIC_TIME_FILTER_CLICK,
            }),
        );
    }, [dispatch]);

    const onChangeFilter = useCallback(() => {
        dispatch(
            loggerActions.logDynamicAction({
                name: EAviaDynamicActionLogActionName.DYNAMIC_TIME_FILTER,
            }),
        );
    }, [dispatch]);

    return (
        <Box between="2" inline>
            <Box>
                <DynamicsNoTransfersFilter />
            </Box>
            <Box>
                <DynamicsBaggageFilter />
            </Box>
            <Box>
                <DynamicsTransferFilter />
            </Box>
            <Box>
                <AviaSearchResultsTimeFilter
                    isActive={filters.time.some(
                        time => time.arrival !== 0 || time.departure !== 0,
                    )}
                    isRoundtrip={isRoundtrip}
                    onClickButton={onClickButton}
                    onChangeFilter={onChangeFilter}
                />
            </Box>
        </Box>
    );
}
