import React, {PureComponent} from 'react';

import IPrice from 'utilities/currency/PriceInterface';
import {CHAR_NBSP, CHAR_EM_DASH} from 'utilities/strings/charCodes';
import {EPriceHintType} from 'projects/avia/components/Dynamics/utilities/getPriceHintType';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/avia-AviaDynamics';

import Price from 'components/Price/Price';
import Flex from 'components/Flex/Flex';
import LinkButton from 'components/LinkButton/LinkButton';
import AviaTrendM from 'icons/illustrations/AviaTrend/AviaTrendM';

import cx from './DynamicsButton.scss';

const mapTypeToHint = {
    [EPriceHintType.EXPENSIVE]: i18nBlock.dynamicsDashHintDashExpensive(),
    [EPriceHintType.ACCEPTABLE]: i18nBlock.dynamicsDashHintDashAcceptable(),
    [EPriceHintType.EMPTY]: i18nBlock.dynamicsDashHintDashEmpty(),
    [EPriceHintType.MINIMAL]: i18nBlock.dynamicsDashHintDashMinimal(),
};

export interface IDynamicsButtonProps {
    onClick(): void;
    priceType: EPriceHintType;

    price?: IPrice;
}

export class DynamicsButton extends PureComponent<IDynamicsButtonProps> {
    private static renderIcon(): React.ReactNode {
        return <AviaTrendM width={24} height={24} />;
    }

    private static renderHint(type: EPriceHintType): React.ReactNode {
        const hint = mapTypeToHint[type];

        return <div className={cx('dynamics-button__hint')}>{hint}</div>;
    }

    private static renderPrice(
        price: IPrice | undefined,
        type: EPriceHintType,
    ): React.ReactNode {
        if (!price) {
            return null;
        }

        return (
            <Flex className={cx('dynamics-button__price')}>
                {`${CHAR_NBSP}${CHAR_NBSP}${CHAR_EM_DASH}${CHAR_NBSP}${CHAR_NBSP}`}
                <div
                    className={cx(
                        'dynamics-button__price-color-box',
                        `_${type}`,
                    )}
                >
                    {i18nBlock.dynamicsDashHintDashPriceDashPreposition()}
                    {CHAR_NBSP}
                    <Price {...price} />
                </div>
            </Flex>
        );
    }

    render(): React.ReactNode {
        const {onClick, priceType, price} = this.props;

        const showPriceDifference = Boolean(price);

        return (
            <Flex
                className={cx('root')}
                {...prepareQaAttributes('avia-desktop-dynamics')}
                alignItems="center"
                between={2}
                inline
                nowrap
                onClick={onClick}
            >
                <Flex className={cx('dynamics-button__content')}>
                    {DynamicsButton.renderIcon()}
                    {DynamicsButton.renderHint(priceType)}
                    {showPriceDifference &&
                        DynamicsButton.renderPrice(price, priceType)}
                </Flex>

                <LinkButton>
                    {i18nBlock.dynamicsDashButtonDashText()}
                </LinkButton>
            </Flex>
        );
    }
}
