import React, {Component} from 'react';
import moment from 'moment';

import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';
import {TExperiments} from 'server/providers/experiments/types';
import {IWithClassName} from 'types/withClassName';

import {ROBOT} from 'utilities/dateUtils/formats';
import getDatesDelta from 'projects/avia/lib/dynamic/getDatesDelta';
import isDateRobot from 'utilities/dateUtils/isDateRobot';
import {CHAR_MINUS} from 'utilities/strings/charCodes';

import * as i18nBlock from 'i18n/avia-AviaDynamics';

import Button from 'components/Button/Button';
import PlusIcon from 'icons/16/Plus';
import MinusIcon from 'icons/16/Minus';

import cx from './DynamicsDeltaButton.scss';

export interface IDynamicsDeltaButtonProps extends IWithClassName {
    isMobile?: boolean;
    searchForm: IAviaParams;
    changeBackwardDate: (backward: string) => void;
    onClickDate: () => void;
    experiments: TExperiments;
}

export class DynamicsDeltaButton extends Component<IDynamicsDeltaButtonProps> {
    private increase = (): void => this.onClick(1);

    private decrease = (): void => this.onClick(-1);

    private onClick(direction: number): void {
        const {searchForm, onClickDate} = this.props;
        const dateForward = moment.utc(searchForm.when);
        const backwardDate = moment
            .utc(searchForm.return_date)
            .add(direction, 'days');

        if (backwardDate.isSameOrAfter(dateForward, 'day')) {
            this.props.changeBackwardDate(backwardDate.format(ROBOT));
        }

        if (onClickDate) {
            onClickDate();
        }
    }

    render(): React.ReactElement | null {
        const {isMobile, searchForm, className} = this.props;

        if (!searchForm.return_date) {
            return null;
        }

        const delta = getDatesDelta(searchForm);

        if (delta === null) {
            return null;
        }

        const buttonSize = isMobile ? 'l' : 'm-inset';

        const {when, return_date: returnDate} = searchForm;

        const startMoment = isDateRobot(when) ? moment(when) : moment();
        const startMonth = startMoment.format('MMM');

        const returnMoment = isDateRobot(returnDate)
            ? moment(returnDate)
            : moment();
        const returnMonth = returnMoment.format('MMM');

        const differentMonth = startMonth !== returnMonth;
        const startDeltaText = startMoment.format(
            `D dd${differentMonth ? ', MMM' : ''}`,
        );
        const deltaText = `${startDeltaText} ${CHAR_MINUS} ${returnMoment.format(
            `D dd, MMM`,
        )}`;

        return (
            <div
                className={cx('dynamicsDelta', className, {
                    dynamicsDelta_mobile: isMobile,
                })}
            >
                <Button
                    size={buttonSize}
                    disabled={delta === 0}
                    onClick={this.decrease}
                    className={cx('control')}
                    icon={<MinusIcon />}
                />
                {isMobile ? (
                    <div className={cx('deltaContainer')}>
                        <div className={cx('deltaText')}>
                            {i18nBlock.dynamicsDashDeltaDashButtonDashText({
                                count: delta,
                            })}
                        </div>

                        <div className={cx('deltaDates')}>{deltaText}</div>
                    </div>
                ) : (
                    <div className={cx('deltaText')}>
                        {i18nBlock.dynamicsDashDeltaDashButtonDashText({
                            count: delta,
                        })}
                    </div>
                )}

                <Button
                    size={buttonSize}
                    onClick={this.increase}
                    className={cx('control')}
                    icon={<PlusIcon />}
                />
            </div>
        );
    }
}
