import React, {FunctionComponent, useCallback} from 'react';
import {parse} from 'query-string';
import {useSelector} from 'react-redux';
import {useHistory} from 'react-router-dom';

import {IDynamicsDay} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import {
    EOneWay,
    IAviaParams,
} from 'server/services/AviaSearchService/types/IAviaParams';

import {filtersHashSelector} from 'selectors/avia/search/filters/filtersHashSelector';

import {isValidDate} from 'utilities/dateUtils';
import {aviaURLs} from 'projects/avia/lib/urls';

import * as i18nBlock from 'i18n/avia-AviaDynamics';

import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import DateLabel from 'components/DateLabel/DateLabel';
import Flex from 'components/Flex/Flex';
import ButtonLink from 'components/ButtonLink/ButtonLink';

interface IAviaDynamicsFooterProps {
    day: IDynamicsDay;
    onLinkClick: (
        date: IDynamicsDay,
        event: React.MouseEvent<HTMLLinkElement>,
    ) => void;
    searchForm: IAviaParams;
    useFiltersFromUrl?: boolean;
}

const AviaDynamicsFooter: FunctionComponent<IAviaDynamicsFooterProps> = ({
    day,
    onLinkClick,
    searchForm,
    useFiltersFromUrl,
}) => {
    const history = useHistory();
    const hash = history.location.hash.slice(1);
    const filtersToHash = useSelector(filtersHashSelector);
    const filters = parse(useFiltersFromUrl ? hash : filtersToHash);
    const {price, dateForward, dateBackward} = day;
    const url = aviaURLs.getSearchResultsUrl(
        {
            ...searchForm,
            oneway:
                searchForm.return_date && isValidDate(searchForm.return_date)
                    ? EOneWay.ROUND_TRIP
                    : EOneWay.ONE_WAY,
            when: day.dateForward.date,
            return_date: day.dateBackward && day.dateBackward.date,
        },
        filters,
    );

    const clickHandler = useCallback(
        (event: React.MouseEvent<HTMLLinkElement>) => {
            onLinkClick(day, event);
        },
        [day, onLinkClick],
    );

    return (
        <Flex alignItems="center" justifyContent="center" textSize="xl">
            <Flex inline alignItems="center" between="10">
                <Flex textWeight="bold" inline between={2}>
                    <DateLabel
                        startDate={dateForward.date}
                        shortFormat={false}
                        endDate={
                            dateBackward ? dateBackward.date : dateForward.date
                        }
                    />

                    <Text weight="bold" size="xl" color="secondary">
                        {dateBackward
                            ? i18nBlock.dynamicsDashFooterDashRoundDashTrip()
                            : i18nBlock.dynamicsDashFooterDashOneDashWay()}
                    </Text>
                </Flex>

                <Flex inline alignItems="center" between="2">
                    {price && (
                        <Text weight="bold" size="xl">
                            <Price
                                value={price.value}
                                currency={price.currency}
                                isRoughly={price.roughly}
                            />
                        </Text>
                    )}

                    <ButtonLink
                        size="m-inset"
                        theme="primary"
                        onClick={clickHandler}
                        url={url}
                    >
                        {i18nBlock.dynamicsDashFooterDashButtonDashText()}
                    </ButtonLink>
                </Flex>
            </Flex>
        </Flex>
    );
};

export default AviaDynamicsFooter;
