import React, {useCallback, useEffect, useMemo} from 'react';
import {useDispatch} from 'react-redux';
import _debounce from 'lodash/debounce';

import {IWithClassName} from 'types/withClassName';
import EAsyncStatus from 'types/common/EAsyncStatus';
import {IDataForRequestDynamic} from 'types/avia/dynamic/IDataForRequestDynamic';
import TValidDataForRequestDynamic from 'types/avia/dynamic/TValidDataForRequestDynamic';
import EAviaDynamicActionLogActionName from 'server/loggers/avia/AviaActionLog/types/EAviaDynamicActionLogActionName';

import {TDynamicsDaysInfo} from 'reducers/avia/aviaPriceIndex/utils/convertPriceIndexDataToDynamicsData';
import {dynamicsRequest} from 'reducers/avia/aviaPriceIndex/dynamics/dynamicsRequest';
import {loggerActions} from 'reducers/avia/aviaLogging/actions';

import {getAviaDynamicsCalendarDates} from 'projects/avia/components/Dynamics/utilities/getAviaDynamicsCalendarDates';
import isDateRobot from 'utilities/dateUtils/isDateRobot';
import {useBoolean} from 'utilities/hooks/useBoolean';
import getDynamicInfo from 'projects/avia/lib/dynamic/getDynamicInfo';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';

import * as i18nBlock from 'i18n/avia-AviaDynamics';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Button from 'components/Button/Button';
import Spinner from 'components/Spinner/Spinner';
import SearchSuccess from './components/SearchSuccess/SearchSuccess';
import CalendarToolbar from './components/CalendarToolbar/CalendarToolbar';
import AviaDynamicsCalendarError from 'projects/avia/components/Dynamics/Calendar/Error/Error';

import cx from './SearchDynamic.scss';

interface ISearchDynamicProps extends IWithClassName {
    isVisible: boolean;
    pricesFromState: TDynamicsDaysInfo;
    searchFormAndFilters: Omit<IDataForRequestDynamic, 'interval'> | null;
    currentRequestParams: TValidDataForRequestDynamic | null;
    status: EAsyncStatus | null;
    isCurrentDayHightlighted?: boolean;

    wrapperClassName?: string;
    hideMoreButton?: boolean;
    onLinkClick?: () => void;
    isFullCalendarForce?: boolean;
    onClickFullCalendar?: () => void;
}

const SearchDynamic: React.FC<ISearchDynamicProps> = ({
    className,
    isVisible,
    pricesFromState,
    searchFormAndFilters,
    currentRequestParams,
    status,

    isCurrentDayHightlighted = true,
    wrapperClassName,
    hideMoreButton,
    onLinkClick,
    isFullCalendarForce,
    onClickFullCalendar: onClickFullCalendarFromProps,
}) => {
    const dispatch = useDispatch();

    const {value: isFullCalendarFromState, setValue: setIsFullCalendar} =
        useBoolean(false);

    const isFullCalendar =
        isFullCalendarForce === undefined
            ? isFullCalendarFromState
            : isFullCalendarForce;

    // Дебаунс для того, чтобы пока закрывается шторка в ней не менялся контент с
    // полного календаря на недельный
    const debouncedSetFullCalendar = useImmutableCallback(
        _debounce(setIsFullCalendar, 300),
    );

    const interval = useMemo(() => {
        const when = searchFormAndFilters?.searchForm.when;

        return when && isDateRobot(when)
            ? getAviaDynamicsCalendarDates(when, isFullCalendar)
            : null;
    }, [isFullCalendar, searchFormAndFilters?.searchForm.when]);

    const {needRequest, actualStatus, requestParams, prices} = getDynamicInfo({
        requestParams:
            searchFormAndFilters && interval
                ? {
                      ...searchFormAndFilters,
                      interval,
                  }
                : null,
        currentRequestParams,
        status,
        prices: pricesFromState,
    });

    const searchIsInProcess = actualStatus === EAsyncStatus.LOADING;
    const searchIsFailed = actualStatus === EAsyncStatus.ERROR;
    const searchIsSuccessful = actualStatus === EAsyncStatus.SUCCESS;

    const loadData = useCallback(() => {
        if (requestParams) {
            dispatch(dynamicsRequest(requestParams));
        }
    }, [dispatch, requestParams]);

    const onClickFullCalendar = useCallback(() => {
        setIsFullCalendar(true);

        dispatch(
            loggerActions.logDynamicAction({
                name: EAviaDynamicActionLogActionName.DYNAMIC_FULL_VIEW,
            }),
        );

        onClickFullCalendarFromProps?.();
    }, [dispatch, setIsFullCalendar, onClickFullCalendarFromProps]);

    useEffect(() => {
        if (needRequest && requestParams) {
            loadData();
        }
    }, [loadData, needRequest, requestParams]);

    useEffect(() => {
        // Сбрасываем состояние на показ недельного интервала при закрытии шторки
        if (!isVisible && isFullCalendar) {
            debouncedSetFullCalendar(false);
        }
    }, [debouncedSetFullCalendar, isFullCalendar, isVisible]);

    return (
        <Flex
            flexDirection="column"
            alignItems="center"
            className={cx('root', className)}
        >
            <CalendarToolbar className={cx('header')} />

            <Box className={cx(wrapperClassName, 'calendar-wrapper')}>
                {searchIsInProcess && (
                    <Box y="10" between="4">
                        <Spinner size="s" />
                        <div className={cx('spinner-text')}>
                            {i18nBlock.dynamicsDashSpinnerDashText()}
                        </div>
                    </Box>
                )}

                {searchIsFailed && (
                    <AviaDynamicsCalendarError
                        className={cx('error')}
                        onRetry={loadData}
                    />
                )}

                {searchIsSuccessful && requestParams && prices && (
                    <SearchSuccess
                        isFullCalendar={isFullCalendarFromState}
                        isVisible={isVisible}
                        requestParams={requestParams}
                        prices={prices}
                        isCurrentDayHightlighted={isCurrentDayHightlighted}
                        onLinkClick={onLinkClick}
                    />
                )}

                {!isFullCalendar && !hideMoreButton && searchIsSuccessful && (
                    <Button
                        onClick={onClickFullCalendar}
                        width="max"
                        className={cx('buttonShowAllCalendar')}
                        size="l"
                    >
                        {i18nBlock.dynamicsDashShowDashAllDashCalendar()}
                    </Button>
                )}
            </Box>
        </Flex>
    );
};

export default React.memo(SearchDynamic);
