import React, {useEffect, useMemo} from 'react';
import {batch, useDispatch, useSelector} from 'react-redux';

import {
    EDynamicsDayStatus,
    IDynamicsCalendarDay,
} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import TValidDataForRequestDynamic from 'types/avia/dynamic/TValidDataForRequestDynamic';
import {EAviaLinkSource} from 'types/avia/url/EAviaLinkSource';

import {setWasOpened} from 'reducers/avia/aviaPriceIndex/dynamics/actions';
import {inlineSearchRequest} from 'reducers/avia/aviaPriceIndex/dynamics/inlineSearchRequest';
import {TDynamicsDaysInfo} from 'reducers/avia/aviaPriceIndex/utils/convertPriceIndexDataToDynamicsData';

import {aviaDynamicsPricesWasOpenedSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesSelectors';

import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import getRoundTripDatesByInterval from 'utilities/dateUtils/getRoundTripDatesByInterval';
import {aviaURLs} from 'projects/avia/lib/urls';
import {
    getDynamicsCalendarDayType,
    getWeekDynamicsCalendarDayType,
} from 'projects/avia/components/Dynamics/utilities/getDynamicsCalendarDayType';
import getDynamicDays from 'projects/avia/components/Dynamics/utilities/getDynamicDays';
import getDatesDelta from 'projects/avia/lib/dynamic/getDatesDelta';
import getFiltersForUrl from 'projects/avia/utilities/getFiltersForUrl';

import AviaDynamicsCalendar from 'projects/avia/components/Dynamics/Calendar/AviaDynamicsCalendar/AviaDynamicsCalendar';

export interface ISearchSuccess {
    isFullCalendar: boolean;
    isVisible: boolean;
    requestParams: TValidDataForRequestDynamic;
    prices: TDynamicsDaysInfo;
    isCurrentDayHightlighted: boolean;

    onLinkClick?: () => void;
}

const SearchSuccess: React.FC<ISearchSuccess> = ({
    isFullCalendar,
    isVisible,
    requestParams,
    prices,
    isCurrentDayHightlighted,
    onLinkClick,
}) => {
    const {searchForm, filters, interval} = requestParams;

    const dispatch = useDispatch();
    const wasOpened = useSelector(aviaDynamicsPricesWasOpenedSelector);

    const delta = getDatesDelta(searchForm);

    const filtersForUrl = useMemo(() => getFiltersForUrl(filters), [filters]);

    const roundTripDatesWithUrl = useMemo(
        () =>
            getRoundTripDatesByInterval(interval, delta).map(date => {
                return {
                    ...date,
                    url: aviaURLs.getSearchResultsUrl(
                        {
                            ...searchForm,
                            when: date.dateForward.date,
                            return_date: date.dateBackward?.date,
                            linkSource: EAviaLinkSource.DYNAMIC,
                        },
                        filtersForUrl,
                    ),
                };
            }),
        [delta, filtersForUrl, interval, searchForm],
    );

    const dynamicsDays = useMemo(
        () => getDynamicDays(roundTripDatesWithUrl, prices),
        [prices, roundTripDatesWithUrl],
    );

    const days: IDynamicsCalendarDay[] = useMemo(() => {
        return dynamicsDays.map((item, index) => ({
            ...item,
            type: isFullCalendar
                ? getWeekDynamicsCalendarDayType(dynamicsDays, index)
                : getDynamicsCalendarDayType(item, dynamicsDays),
        }));
    }, [dynamicsDays, isFullCalendar]);

    const inlineRequest = useImmutableCallback(
        (dateForward: string, dateBackward: Nullable<string>) =>
            dispatch(inlineSearchRequest(dateForward, dateBackward)),
    );

    // Запуск инлайн поисков для первого экрана динамики
    // Не запускаем на изменение интервала/фильтров, т.к. будет излишнее кол-во запросов
    useEffect(() => {
        if (isVisible && !wasOpened && dynamicsDays.length && !isFullCalendar) {
            batch(() => {
                dynamicsDays
                    .filter(
                        day => day.status === EDynamicsDayStatus.SHOULD_SEARCH,
                    )
                    .forEach(({dateForward, dateBackward}) =>
                        inlineRequest(
                            dateForward.date,
                            dateBackward && dateBackward.date,
                        ),
                    );
                dispatch(setWasOpened(true));
            });
        }
    }, [
        dynamicsDays,
        inlineRequest,
        isFullCalendar,
        isVisible,
        wasOpened,
        dispatch,
    ]);

    return (
        <AviaDynamicsCalendar
            inlineSearchRequest={inlineRequest}
            days={days}
            isCurrentDayHightlighted={isCurrentDayHightlighted}
            onLinkClick={onLinkClick}
        />
    );
};

export default React.memo(SearchSuccess);
