import React, {useMemo} from 'react';
import {useDispatch} from 'react-redux';

import {IDynamicsCalendarDay} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import TValidDataForRequestDynamic from 'types/avia/dynamic/TValidDataForRequestDynamic';

import {inlineSearchRequest} from 'reducers/avia/aviaPriceIndex/dynamics/inlineSearchRequest';
import {TDynamicsDaysInfo} from 'reducers/avia/aviaPriceIndex/utils/convertPriceIndexDataToDynamicsData';

import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import getRoundTripDatesByInterval from 'utilities/dateUtils/getRoundTripDatesByInterval';
import {aviaURLs} from 'projects/avia/lib/urls';
import getDatesDelta from 'projects/avia/lib/dynamic/getDatesDelta';
import getFiltersForUrl from 'projects/avia/utilities/getFiltersForUrl';
import getDynamicDays from 'projects/avia/components/Dynamics/utilities/getDynamicDays';
import {getWeekDynamicsCalendarDayType} from 'projects/avia/components/Dynamics/utilities/getDynamicsCalendarDayType';

import AviaDynamicsCalendar from 'projects/avia/components/Dynamics/Calendar/AviaDynamicsCalendar/AviaDynamicsCalendar';

export interface ISearchSuccess {
    requestParams: TValidDataForRequestDynamic;
    prices: TDynamicsDaysInfo;

    onLinkClick?: () => void;
}

const SearchSuccess: React.FC<ISearchSuccess> = ({
    requestParams,
    prices,

    onLinkClick,
}) => {
    const {searchForm, filters, interval} = requestParams;

    const dispatch = useDispatch();

    const delta = getDatesDelta(searchForm);

    const filtersForUrl = useMemo(() => getFiltersForUrl(filters), [filters]);

    const roundTripDatesWithUrl = useMemo(
        () =>
            getRoundTripDatesByInterval(interval, delta).map(date => {
                return {
                    ...date,
                    url: aviaURLs.getSearchResultsUrl(
                        {
                            ...searchForm,
                            when: date.dateForward.date,
                            return_date: date.dateBackward?.date,
                        },
                        filtersForUrl,
                    ),
                };
            }),
        [delta, filtersForUrl, interval, searchForm],
    );

    const dynamicsDays = useMemo(
        () => getDynamicDays(roundTripDatesWithUrl, prices),
        [prices, roundTripDatesWithUrl],
    );

    const days: IDynamicsCalendarDay[] = useMemo(() => {
        return dynamicsDays.map((item, index) => ({
            ...item,
            type: getWeekDynamicsCalendarDayType(dynamicsDays, index),
        }));
    }, [dynamicsDays]);

    const inlineRequest = useImmutableCallback(
        (dateForward: string, dateBackward: Nullable<string>) =>
            dispatch(
                inlineSearchRequest(
                    dateForward,
                    dateBackward,
                    searchForm,
                    filters,
                ),
            ),
    );

    return (
        <AviaDynamicsCalendar
            inlineSearchRequest={inlineRequest}
            days={days}
            isCurrentDayHightlighted
            onLinkClick={onLinkClick}
        />
    );
};

export default React.memo(SearchSuccess);
