import React, {useCallback, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useHistory} from 'react-router-dom';

import EAsyncStatus from 'types/common/EAsyncStatus';

import {dynamicsRequest} from 'reducers/avia/aviaPriceIndex/dynamics/dynamicsRequest';
import {
    getInitialState,
    IAviaSearchResultsFilters,
} from 'reducers/avia/search/results/filters/reducer';

import {aviaDynamicsPricesFinalPriceSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesFinalPriceSelector';
import {aviaDynamicsPricesStatusSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesSelectors';
import {aviaDynamicsPricesSearchFormSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesSearchFormSelector';
import aviaDynamicsPricesCurrentRequestParamsSelector from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesCurrentRequestParamsSelector';

import {hashToFilterValues} from 'projects/avia/lib/search/filters/converters';
import {getAviaDynamicsCalendarDates} from 'projects/avia/components/Dynamics/utilities/getAviaDynamicsCalendarDates';
import isDateRobot from 'utilities/dateUtils/isDateRobot';
import getDynamicInfo from 'projects/avia/lib/dynamic/getDynamicInfo';

import * as i18nBlock from 'i18n/avia-AviaDynamics';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Spinner from 'components/Spinner/Spinner';
import AviaDynamicsCalendarError from 'projects/avia/components/Dynamics/Calendar/Error/Error';
import StandaloneFilters from 'projects/avia/components/Dynamics/StandaloneFilters/StandaloneFilters';
import DynamicsDeltaButtonConnected from 'projects/avia/components/Dynamics/DynamicsDeltaButton/DynamicsDeltaButtonConnected';
import SearchSuccess from 'projects/avia/components/Dynamics/StandaloneDynamic/SearchSuccess';

import cx from './StandaloneDynamic.scss';

const FILTERS_DEFAULT = getInitialState();

interface IStandaloneCalendarContentProps {
    wrapperClassName?: string;
    onSearch: () => void;
    onFilterChange: () => void;
}

const StandaloneCalendarContent: React.FC<IStandaloneCalendarContentProps> = ({
    wrapperClassName,
    onSearch,
    onFilterChange,
}) => {
    const history = useHistory();
    const dispatch = useDispatch();
    const hash = history.location.hash.slice(1);
    const filters: IAviaSearchResultsFilters = useMemo(
        () => ({...FILTERS_DEFAULT, ...hashToFilterValues(hash)}),
        [hash],
    );

    const pricesFromState = useSelector(aviaDynamicsPricesFinalPriceSelector);
    const searchForm = useSelector(aviaDynamicsPricesSearchFormSelector);
    const currentRequestParams = useSelector(
        aviaDynamicsPricesCurrentRequestParamsSelector,
    );
    const status = useSelector(aviaDynamicsPricesStatusSelector);
    const interval = useMemo(() => {
        const when = searchForm.when;

        return when && isDateRobot(when)
            ? getAviaDynamicsCalendarDates(when, false)
            : null;
    }, [searchForm.when]);

    const {needRequest, actualStatus, requestParams, prices} = getDynamicInfo({
        requestParams: interval
            ? {
                  searchForm,
                  filters,
                  interval,
              }
            : null,
        currentRequestParams,
        status,
        prices: pricesFromState,
    });

    const searchIsInProcess = actualStatus === EAsyncStatus.LOADING;
    const searchIsFailed = actualStatus === EAsyncStatus.ERROR;
    const searchIsSuccessful = actualStatus === EAsyncStatus.SUCCESS;

    const loadData = useCallback(() => {
        if (requestParams) {
            dispatch(dynamicsRequest(requestParams));
        }
    }, [dispatch, requestParams]);

    useEffect(() => {
        if (needRequest && requestParams) {
            loadData();
        }
    }, [loadData, needRequest, requestParams]);

    return (
        <Flex flexDirection="column" alignItems="center">
            <Flex
                className={cx('header')}
                flexDirection="column"
                between="3"
                y="3"
            >
                <DynamicsDeltaButtonConnected isMobile />
                <StandaloneFilters onChange={onFilterChange} />
            </Flex>

            <Box
                className={cx(wrapperClassName, 'calendar-wrapper')}
                between="8"
            >
                {searchIsInProcess && (
                    <Box y="10" between="4">
                        <Spinner size="s" />
                        <div className={cx('spinner-text')}>
                            {i18nBlock.dynamicsDashSpinnerDashText()}
                        </div>
                    </Box>
                )}

                {searchIsFailed && (
                    <AviaDynamicsCalendarError onRetry={loadData} />
                )}

                {searchIsSuccessful && requestParams && prices && (
                    <SearchSuccess
                        requestParams={requestParams}
                        prices={prices}
                        onLinkClick={onSearch}
                    />
                )}
            </Box>
        </Flex>
    );
};

export default StandaloneCalendarContent;
