import React from 'react';

import {IStandaloneDynamicsFilterType} from 'projects/avia/components/Dynamics/StandaloneFilters/type';

import {
    EAviaSearchTransferFilter,
    IAviaSearchTimeFilter,
} from 'reducers/avia/search/results/filters/reducer';

import {isEnabled} from 'utilities/flags/flags';
import {useBoolean} from 'utilities/hooks/useBoolean';

import * as keyset from 'i18n/avia-search';

import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import FilterIcon from 'icons/16/Filter';
import Separator from 'components/Separator/Separator';
import CheckButton from 'components/CheckButton/CheckButton';
import Intersperse from 'components/Intersperse/Intersperse';
import Raskrivashka from 'components/Raskrivashka/Raskrivashka';
import ModalWithBackButton from 'components/ModalWithBackButton/ModalWithBackButton';
import {AviaTimeFilterPopup} from 'projects/avia/components/AviaSearchResultsTimeFilter/AviaTimeFilterPopup';
import {AviaTransferFilterPopup} from 'projects/avia/components/AviaSearchResultsTransferFilter/AviaTransferFilterPopup/AviaTransferFilterPopup';

import cx from './StandaloneFilters.scss';

const MobileStandaloneFilters: React.FC<IStandaloneDynamicsFilterType> = ({
    time,
    baggage,
    transfer,
    handleTimeChange,
    handleBaggageChange,
    handleTransfersChange,
    handleNoTransfersChange,
}) => {
    const {
        value: filtersModalIsVisible,
        setFalse: closeFiltersModal,
        setTrue: openFiltersModal,
    } = useBoolean(false);

    const transfersFilterIsActive = transfer.value !== 0;
    const timeFilterIsActive = time.arrival !== 0 || time.departure !== 0;

    return (
        <Flex inline nowrap between="2">
            <CheckButton
                width="max"
                size="l"
                checked={isEnabled(
                    transfer.value,
                    EAviaSearchTransferFilter.NO_TRANSFERS,
                )}
                onClick={handleNoTransfersChange}
            >
                <Text size="s">{keyset.filterDotNoDashTransfers()}</Text>
            </CheckButton>

            <CheckButton
                width="max"
                size="l"
                checked={baggage.enabled}
                onClick={handleBaggageChange}
            >
                <Text size="s">{keyset.filterDotWithDashBaggage()}</Text>
            </CheckButton>

            <CheckButton
                icon={<FilterIcon />}
                checked={transfersFilterIsActive || timeFilterIsActive}
                onClick={openFiltersModal}
                size="l"
            />

            <ModalWithBackButton
                isVisible={filtersModalIsVisible}
                onClose={closeFiltersModal}
                canRenderBackButton
                fullScreen
                preventBodyScroll={filtersModalIsVisible}
                containerClassName={cx('modal-container')}
            >
                <Separator />
                <Intersperse separator={<Separator />}>
                    <Raskrivashka
                        contentIsInvolved={transfersFilterIsActive}
                        label={keyset.filterDotTransfersDashPlaceholder()}
                        initialOpened
                    >
                        <Box x="4">
                            <AviaTransferFilterPopup
                                hideRange
                                hideTransferAirports
                                transfersValue={transfer.value}
                                onChangeTransfers={handleTransfersChange}
                            />
                        </Box>
                    </Raskrivashka>

                    <Raskrivashka
                        contentIsInvolved={timeFilterIsActive}
                        label={keyset.filterDotTimeDotTitle()}
                        initialOpened
                    >
                        <Box x="4" between="5">
                            <AviaTimeFilterPopup
                                segmentIdx={0}
                                disabled={
                                    {
                                        arrival: 0,
                                        departure: 0,
                                    } as IAviaSearchTimeFilter
                                }
                                value={
                                    time ||
                                    ({
                                        arrival: 0,
                                        departure: 0,
                                    } as IAviaSearchTimeFilter)
                                }
                                onChange={handleTimeChange}
                            />
                        </Box>
                    </Raskrivashka>
                </Intersperse>
            </ModalWithBackButton>
        </Flex>
    );
};

export default MobileStandaloneFilters;
