import React, {useCallback, useEffect, useReducer, useRef} from 'react';
import {useHistory} from 'react-router-dom';

import filtersReducer, {
    getInitialState,
    IAviaSearchTimeFilter,
    EAviaSearchTransferFilter,
} from 'reducers/avia/search/results/filters/reducer';
import {
    setTransferFilter,
    setBaggageFilter,
    setFilterValues,
    setTimeFilter,
} from 'reducers/avia/search/results/filters/actions';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {disable, enable, Flag, isEnabled} from 'utilities/flags/flags';
import {getAppliedAviaFilters} from 'projects/avia/lib/search/filters/getAppliedFilters';
import {
    filterValuesToHash,
    hashToFilterValues,
} from 'projects/avia/lib/search/filters/converters';

import DesktopStandaloneFilters from 'projects/avia/components/Dynamics/StandaloneFilters/DesktopStandaloneFilters';
import MobileStandaloneFilters from 'projects/avia/components/Dynamics/StandaloneFilters/MobileStandaloneFilters';

interface IStandaloneFiltersProps {
    onChange: () => void;
}

const StandaloneFilters: React.FC<IStandaloneFiltersProps> = ({onChange}) => {
    const history = useHistory();
    const mountedRef = useRef<boolean>(false);
    const [filters, dispatch] = useReducer(filtersReducer, getInitialState());
    const {transfer, baggage, time} = filters;
    const deviceType = useDeviceType();
    const Filters = deviceType.isDesktop
        ? DesktopStandaloneFilters
        : MobileStandaloneFilters;

    useEffect(() => {
        const hash = history.location.hash.slice(1);
        const filterValues = hashToFilterValues(hash);

        dispatch(setFilterValues(filterValues));
    }, [history, dispatch]);

    useEffect(() => {
        if (!mountedRef.current) {
            mountedRef.current = true;

            return;
        }

        const appliedFilters = getAppliedAviaFilters(filters);
        const filtersHash = filterValuesToHash(appliedFilters);

        history.replace({
            hash: filtersHash,
            pathname: history.location.pathname,
            search: history.location.search,
            state: history.location.state,
        });
    }, [history, filters, mountedRef]);

    const handleNoTransfersChange = useCallback(() => {
        dispatch(
            setTransferFilter(
                isEnabled(
                    transfer.value,
                    EAviaSearchTransferFilter.NO_TRANSFERS,
                )
                    ? disable(
                          transfer.value,
                          EAviaSearchTransferFilter.NO_TRANSFERS,
                      )
                    : enable(
                          transfer.value,
                          EAviaSearchTransferFilter.NO_TRANSFERS,
                      ),
            ),
        );
        onChange();
    }, [dispatch, transfer, onChange]);

    const handleBaggageChange = useCallback(() => {
        dispatch(setBaggageFilter(!baggage.enabled));
        onChange();
    }, [dispatch, baggage, onChange]);

    const handleTransfersChange = useCallback(
        (value: Flag<EAviaSearchTransferFilter>) => {
            dispatch(setTransferFilter(value));
            onChange();
        },
        [dispatch, onChange],
    );

    const handleTimeChange = useCallback(
        (segmentVal: IAviaSearchTimeFilter, segmentIdx: number) => {
            const nextFilterValue = time.slice();

            nextFilterValue[segmentIdx] = segmentVal;

            dispatch(setTimeFilter(nextFilterValue));
            onChange();
        },
        [dispatch, time, onChange],
    );

    return (
        <Filters
            time={time[0]}
            baggage={baggage}
            transfer={transfer}
            handleTimeChange={handleTimeChange}
            handleBaggageChange={handleBaggageChange}
            handleTransfersChange={handleTransfersChange}
            handleNoTransfersChange={handleNoTransfersChange}
        />
    );
};

export default StandaloneFilters;
