import React from 'react';

import {
    EDynamicsDayStatus,
    IDynamicsCalendarDay,
} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import {TInlineSearchRequest} from 'projects/avia/components/Dynamics/types/TInlineSearchRequest';
import {IWithClassName} from 'types/withClassName';

import {EPriceHintType} from 'projects/avia/components/Dynamics/utilities/getPriceHintType';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {CHAR_EM_DASH} from 'utilities/strings/charCodes';

import {
    dynamicDashPriceDashNoDashTickets,
    dynamicsDashError,
} from 'i18n/avia-AviaDynamics';

import Spinner from 'components/Spinner/Spinner';
import Price from 'components/Price/Price';
import Search from 'icons/24/Search';
import CloseForCircleBgRefresh16 from 'icons/16/CloseForCircleBgRefresh16';

import cx from './DayButton.scss';

export interface IDayButtonProps extends IWithClassName {
    day: IDynamicsCalendarDay;
    inlineSearchRequest: TInlineSearchRequest;
    isCurrent: boolean; // Признак что день является контекстом поиска

    onLinkClick?: () => void; // Колбэк на клик по ссылке на поиск
}

const DayButton: React.FC<IDayButtonProps> = ({
    day,
    inlineSearchRequest,
    isCurrent,

    onLinkClick,
    className,
}) => {
    const {status, price, type, dateForward, dateBackward, url} = day;

    const isMinimalPrice = type === EPriceHintType.MINIMAL;
    const isSearchLink = !isCurrent && price && !price.roughly;

    let content: React.ReactElement | string;

    const clickHandler = useImmutableCallback(() => {
        if (
            status !== EDynamicsDayStatus.NO_DATA &&
            status !== EDynamicsDayStatus.SEARCHING &&
            !isCurrent
        ) {
            inlineSearchRequest(dateForward.date, dateBackward?.date || null);
        }
    });

    if (dateBackward && dateForward.date > dateBackward.date) {
        return (
            <span className={cx('root', className, 'root_impossible')}>
                {CHAR_EM_DASH}
            </span>
        );
    }

    switch (status) {
        case EDynamicsDayStatus.NO_DATA:
            content = dynamicDashPriceDashNoDashTickets();

            break;
        case EDynamicsDayStatus.HAS_DATA:
        case EDynamicsDayStatus.SHOULD_SEARCH:
            content = price ? (
                <Price
                    {...price}
                    prefix={price.roughly ? '≈' : undefined}
                    prefixDelimiter=" "
                    className={cx('price')}
                />
            ) : (
                <Search className={cx('searchIcon')} />
            );

            break;
        case EDynamicsDayStatus.SEARCHING:
            content = <Spinner size="s" />;

            break;
        case EDynamicsDayStatus.WAS_ERROR:
            content = (
                <>
                    <span>{dynamicsDashError()}</span>
                    <CloseForCircleBgRefresh16 />
                </>
            );

            break;
    }

    return (
        <a
            className={cx('root', className, {
                root_isMinimalPrice: isMinimalPrice,
                root_noData: status === EDynamicsDayStatus.NO_DATA,
                root_error: status === EDynamicsDayStatus.WAS_ERROR,
                root_isCurrent: isCurrent,
            })}
            href={isSearchLink ? url : undefined}
            onClick={isSearchLink ? onLinkClick : clickHandler}
            target="_blank"
            rel="noopener noreferrer"
        >
            {content}
        </a>
    );
};

export default React.memo(DayButton);
