import React, {useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {TInlineSearchRequest} from 'projects/avia/components/Dynamics/types/TInlineSearchRequest';
import EAsyncStatus from 'types/common/EAsyncStatus';

import {dynamicsRequest} from 'reducers/avia/aviaPriceIndex/dynamics/dynamicsRequest';

import {aviaDynamicsPricesStatusSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesSelectors';
import {aviaDynamicsPricesFinalPriceSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesFinalPriceSelector';
import {aviaDynamicsPricesRequestParamsSelector} from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesRequestParamsSelector';
import aviaDynamicsPricesCurrentRequestParamsSelector from 'selectors/avia/dynamicsPrices/aviaDynamicsPricesCurrentRequestParamsSelector';

import {getAviaDynamicsCalendarYearDates} from 'projects/avia/components/Dynamics/utilities/getAviaDynamicsCalendarDates';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import getDynamicInfo from 'projects/avia/lib/dynamic/getDynamicInfo';

import SearchSuccess from 'projects/avia/components/Dynamics/Table/OneWayTable/SearchSuccess';
import Loading from 'projects/avia/components/Dynamics/Table/Loading/Loading';
import Error from 'projects/avia/components/Dynamics/Table/Error/Error';

import cx from './OneWayTable.scss';

export interface IOneWayTableProps {
    inlineSearchRequest: TInlineSearchRequest;
    onLinkClick: () => void | undefined; // Колбэк на клик по ссылке на поиск
}

const OneWayTable: React.FC<IOneWayTableProps> = ({
    inlineSearchRequest,
    onLinkClick,
}) => {
    const dispatch = useDispatch();
    const pricesFromState = useSelector(aviaDynamicsPricesFinalPriceSelector);
    const searchFormAndFilters = useSelector(
        aviaDynamicsPricesRequestParamsSelector,
    );
    const currentRequestParams = useSelector(
        aviaDynamicsPricesCurrentRequestParamsSelector,
    );
    const status = useSelector(aviaDynamicsPricesStatusSelector);

    const interval = useMemo(() => getAviaDynamicsCalendarYearDates(), []);

    const {needRequest, actualStatus, requestParams, prices} = getDynamicInfo({
        requestParams:
            searchFormAndFilters && interval
                ? {
                      ...searchFormAndFilters,
                      interval,
                  }
                : null,
        currentRequestParams,
        status,
        prices: pricesFromState,
    });

    const searchIsInProcess = actualStatus === EAsyncStatus.LOADING;
    const searchIsFailed = actualStatus === EAsyncStatus.ERROR;
    const searchIsSuccessful = actualStatus === EAsyncStatus.SUCCESS;

    const loadData = useImmutableCallback(() => {
        if (requestParams) {
            dispatch(dynamicsRequest(requestParams));
        }
    });

    useEffect(() => {
        if (needRequest && requestParams) {
            loadData();
        }
    }, [loadData, needRequest, requestParams]);

    return (
        <div className={cx('root')}>
            {searchIsInProcess && <Loading type="oneWay" />}

            {searchIsFailed && <Error type="oneWay" loadData={loadData} />}

            {searchIsSuccessful && requestParams && prices && (
                <SearchSuccess
                    requestParams={requestParams}
                    prices={prices}
                    inlineSearchRequest={inlineSearchRequest}
                    onLinkClick={onLinkClick}
                />
            )}
        </div>
    );
};

export default React.memo(OneWayTable);
