import React, {useMemo, useState} from 'react';

import {IDynamicsCalendarDay} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import TValidDataForRequestDynamic from 'types/avia/dynamic/TValidDataForRequestDynamic';
import {TInlineSearchRequest} from 'projects/avia/components/Dynamics/types/TInlineSearchRequest';
import {EAviaLinkSource} from 'types/avia/url/EAviaLinkSource';

import {TDynamicsDaysInfo} from 'reducers/avia/aviaPriceIndex/utils/convertPriceIndexDataToDynamicsData';

import getRoundTripDatesByInterval from 'utilities/dateUtils/getRoundTripDatesByInterval';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import getDynamicDays from 'projects/avia/components/Dynamics/utilities/getDynamicDays';
import {
    getWeekDynamicsCalendarDayType,
    IPriceDay,
} from 'projects/avia/components/Dynamics/utilities/getDynamicsCalendarDayType';
import {aviaURLs} from 'projects/avia/lib/urls';
import getFiltersForUrl from 'projects/avia/utilities/getFiltersForUrl';

import RoundButton from 'components/RoundButton/RoundButton';
import ArrowLeftIcon from 'icons/16/ArrowLeft';
import ArrowRightIcon from 'icons/16/ArrowRight';
import Day from 'projects/avia/components/Dynamics/Table/Day/Day';
import DayButton from 'projects/avia/components/Dynamics/Table/DayButton/DayButton';
import {
    DATES_RANGE_ADDITIONAL,
    FULL_WIDTH_DAY,
    ONE_WAY_DATES_RANGE,
} from 'projects/avia/components/Dynamics/Table/const';

import cx from './SearchSuccess.scss';

export interface ISearchSuccessProps {
    requestParams: TValidDataForRequestDynamic;
    prices: TDynamicsDaysInfo;
    inlineSearchRequest: TInlineSearchRequest;
    onLinkClick: (() => void) | undefined;
}

const SearchSuccess: React.FC<ISearchSuccessProps> = ({
    requestParams,
    prices,
    inlineSearchRequest,
    onLinkClick,
}) => {
    const {searchForm, interval, filters} = requestParams;
    const {when} = searchForm;

    const filtersForUrl = useMemo(() => getFiltersForUrl(filters), [filters]);
    const dates = useMemo(
        () =>
            getRoundTripDatesByInterval(interval, null).map((date, index) => ({
                ...date,
                index,
            })),
        [interval],
    );
    const datePrices: IPriceDay[] = useMemo(
        () =>
            dates.map(({dateForward: {date}}) => ({
                price: prices[date]?.price || null,
            })),
        [dates, prices],
    );
    const maxDaysIndex = dates.length - 1;
    const maxLeftIndex = maxDaysIndex - ONE_WAY_DATES_RANGE * 2;

    const [leftIndex, setLeftIndex] = useState(
        Math.max(
            dates.findIndex(date => date.dateForward.date === when) -
                ONE_WAY_DATES_RANGE,
            0,
        ),
    );
    const rightIndex = leftIndex + ONE_WAY_DATES_RANGE * 2;

    const onClickLeftButton = useImmutableCallback(() => {
        if (leftIndex > 0) {
            setLeftIndex(leftIndex - 1);
        }
    });
    const onClickRightButton = useImmutableCallback(() => {
        if (leftIndex < maxLeftIndex) {
            setLeftIndex(leftIndex + 1);
        }
    });

    // Окно дат для рендера
    const startIndexDaysForRender = Math.max(
        leftIndex - DATES_RANGE_ADDITIONAL,
        0,
    );
    const endIndexDaysForRender = Math.min(
        rightIndex + DATES_RANGE_ADDITIONAL,
        maxDaysIndex,
    );
    const widthFakeBlock = startIndexDaysForRender * FULL_WIDTH_DAY;
    const transformX = -leftIndex * FULL_WIDTH_DAY;

    // Дни для рендера
    const dynamicDays = getDynamicDays(
        dates.slice(startIndexDaysForRender, endIndexDaysForRender + 1),
        prices,
    );
    const days: IDynamicsCalendarDay[] = dynamicDays.map(date => {
        return {
            ...date,
            type: getWeekDynamicsCalendarDayType(datePrices, date.index),
            url: aviaURLs.getSearchResultsUrl(
                {
                    ...searchForm,
                    when: date.dateForward.date,
                    return_date: undefined,
                    linkSource: EAviaLinkSource.DYNAMIC,
                },
                filtersForUrl,
            ),
        };
    });

    return (
        <>
            <RoundButton
                className={cx('leftButton')}
                size="s"
                theme="white"
                icon={<ArrowLeftIcon />}
                disabled={leftIndex === 0}
                onClick={onClickLeftButton}
            />
            <RoundButton
                className={cx('rightButton')}
                size="s"
                theme="white"
                icon={<ArrowRightIcon />}
                disabled={leftIndex === maxLeftIndex}
                onClick={onClickRightButton}
            />
            <div className={cx('view')}>
                <div
                    className={cx('content')}
                    style={{transform: `translateX(${transformX}px)`}}
                >
                    <div className={cx('row')}>
                        <div
                            className={cx('fakeBlock')}
                            style={{width: widthFakeBlock}}
                        />
                        {days.map(day => (
                            <Day
                                className={cx('th')}
                                key={day.dateForward.date}
                                date={day.dateForward.date}
                                isHoliday={day.dateForward.isHoliday}
                            />
                        ))}
                    </div>
                    <div className={cx('row')}>
                        <div
                            className={cx('fakeBlock')}
                            style={{width: widthFakeBlock}}
                        />
                        {days.map(day => (
                            <DayButton
                                className={cx('td')}
                                key={day.dateForward.date}
                                day={day}
                                isCurrent={day.dateForward.date === when}
                                inlineSearchRequest={inlineSearchRequest}
                                onLinkClick={onLinkClick}
                            />
                        ))}
                    </div>
                </div>
            </div>
        </>
    );
};

export default React.memo(SearchSuccess);
