import React, {useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {TInlineSearchRequest} from 'projects/avia/components/Dynamics/types/TInlineSearchRequest';
import EAsyncStatus from 'types/common/EAsyncStatus';
import ITableDynamicCalendarDate from 'projects/avia/components/Dynamics/types/ITableDynamicCalendarDate';
import TDateRobot from 'types/common/date/TDateRobot';

import tableDynamicRequest from 'reducers/avia/aviaPriceIndex/tableDynamic/tableDynamicRequest';

import tableDynamicSearchFormSelector from 'selectors/avia/tableDynamic/tableDynamicSearchFormSelector';
import tableDynamicFiltersSelector from 'selectors/avia/tableDynamic/tableDynamicFiltersSelector';
import tableDynamicCurrentRequestParamsSelector from 'selectors/avia/tableDynamic/tableDynamicCurrentRequestParamsSelector';
import {tableDynamicStatusSelector} from 'selectors/avia/tableDynamic/tableDynamicSeletors';
import tableDynamicFinalPriceSelector from 'selectors/avia/tableDynamic/tableDynamicFinalPriceSelector';

import getTableDynamicInfo from 'projects/avia/lib/dynamic/getTableDynamicInfo';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {getAviaDynamicsCalendarYearDates} from 'projects/avia/components/Dynamics/utilities/getAviaDynamicsCalendarDates';
import getRoundTripDatesByInterval from 'utilities/dateUtils/getRoundTripDatesByInterval';
import {IGetIntervalsForRequestTableDynamicCoords} from 'projects/avia/lib/dynamic/getIntervalsForRequestTableDynamic';
import getNearestIndexes from 'projects/avia/lib/dynamic/getNearestIndexes';

import Loading from 'projects/avia/components/Dynamics/Table/Loading/Loading';
import Error from 'projects/avia/components/Dynamics/Table/Error/Error';
import SearchSuccess from 'projects/avia/components/Dynamics/Table/RoundTripTable/SearchSuccess';
import {
    DATES_RANGE_ADDITIONAL,
    ROUND_TRIP_DATES_RANGE_HORIZONTAL,
    ROUND_TRIP_DATES_RANGE_VERTICAL,
} from 'projects/avia/components/Dynamics/Table/const';

import cx from './RoundTripTable.scss';

export interface IRoundTripTableProps {
    inlineSearchRequest: TInlineSearchRequest;
    searchWhen: TDateRobot; // Дата "туда" из контекста поиска
    searchReturnDate: TDateRobot; // Дата "обратно" из контекста поиска
    onLinkClick: () => void | undefined; // Колбэк на клик по ссылке на поиск
}

const RoundTripTable: React.FC<IRoundTripTableProps> = ({
    inlineSearchRequest,
    searchWhen,
    searchReturnDate,
    onLinkClick,
}) => {
    const dispatch = useDispatch();
    const searchForm = useSelector(tableDynamicSearchFormSelector);
    const filters = useSelector(tableDynamicFiltersSelector);
    const currentRequestParams = useSelector(
        tableDynamicCurrentRequestParamsSelector,
    );
    const status = useSelector(tableDynamicStatusSelector);
    const pricesFromState = useSelector(tableDynamicFinalPriceSelector);

    const {requestParams, needRequest, actualStatus, prices} =
        getTableDynamicInfo({
            requestParams: {
                searchForm,
                filters,
            },
            currentRequestParams,
            status,
            prices: pricesFromState,
        });

    const searchIsInProcess = actualStatus === EAsyncStatus.LOADING;
    const searchIsFailed = actualStatus === EAsyncStatus.ERROR;
    const searchIsSuccessful = actualStatus === EAsyncStatus.SUCCESS;

    const interval = useMemo(() => getAviaDynamicsCalendarYearDates(), []);
    const dates: ITableDynamicCalendarDate[] = useMemo(
        () =>
            getRoundTripDatesByInterval(interval, null).map(
                ({dateForward}, index) => ({...dateForward, index}),
            ),
        [interval],
    );
    const forwardDates: ITableDynamicCalendarDate[] = dates;
    const backwardDates: ITableDynamicCalendarDate[] = dates;

    // Определяем окно в таблице, для которого пойдем за данными о ценах в ручку динамики
    const ranges: IGetIntervalsForRequestTableDynamicCoords | undefined =
        useMemo(() => {
            if (!requestParams) {
                return;
            }

            const {
                searchForm: {when, return_date: returnDate},
            } = requestParams;
            const forwardDateIndex = forwardDates.findIndex(
                ({date}) => date === when,
            );
            const backwardDateIndex = backwardDates.findIndex(
                ({date}) => date === returnDate,
            );

            const {start: forwardStart, end: forwardEnd} = getNearestIndexes(
                forwardDates,
                forwardDateIndex,
                ROUND_TRIP_DATES_RANGE_HORIZONTAL + DATES_RANGE_ADDITIONAL + 2,
            );

            const {start: backwardStart, end: backwardEnd} = getNearestIndexes(
                backwardDates,
                backwardDateIndex,
                ROUND_TRIP_DATES_RANGE_VERTICAL + DATES_RANGE_ADDITIONAL + 2,
            );

            return {
                leftRange: backwardDateIndex - backwardStart,
                rightRange: backwardEnd - backwardDateIndex,
                topRange: forwardDateIndex - forwardStart,
                bottomRange: forwardEnd - forwardDateIndex,
            };
        }, [backwardDates, forwardDates, requestParams]);

    const loadData = useImmutableCallback(() => {
        if (requestParams && ranges) {
            dispatch(
                tableDynamicRequest({
                    requestParams,
                    ...ranges,
                }),
            );
        }
    });

    useEffect(() => {
        if (needRequest && requestParams && ranges) {
            loadData();
        }
    }, [loadData, needRequest, ranges, requestParams]);

    return (
        <div className={cx('root')}>
            {searchIsInProcess && <Loading type="roundTrip" />}

            {searchIsFailed && <Error type="roundTrip" loadData={loadData} />}

            {searchIsSuccessful && requestParams && prices && (
                <SearchSuccess
                    requestParams={requestParams}
                    prices={prices}
                    inlineSearchRequest={inlineSearchRequest}
                    onLinkClick={onLinkClick}
                    datesForward={forwardDates}
                    datesBackward={backwardDates}
                    searchWhen={searchWhen}
                    searchReturnDate={searchReturnDate}
                />
            )}
        </div>
    );
};

export default React.memo(RoundTripTable);
