import React, {useEffect, useMemo, useState} from 'react';
import {batch} from 'react-redux';

import {TInlineSearchRequest} from 'projects/avia/components/Dynamics/types/TInlineSearchRequest';
import TValidDataForRequestTableDynamic from 'types/avia/dynamic/TValidDataForRequestTableDynamic';
import {
    EDynamicsDayStatus,
    IDynamicsTableCalendarDay,
} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import ITableDynamicCalendarDate from 'projects/avia/components/Dynamics/types/ITableDynamicCalendarDate';
import TDateRobot from 'types/common/date/TDateRobot';
import {EAviaLinkSource} from 'types/avia/url/EAviaLinkSource';

import {TDynamicTableDaysInfo} from 'reducers/avia/aviaPriceIndex/utils/convertPriceIndexDataToDynamicsData';

import getFiltersForUrl from 'projects/avia/utilities/getFiltersForUrl';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import getDynamicDayForTable from 'projects/avia/components/Dynamics/utilities/getDynamicDayForTable';
import {aviaURLs} from 'projects/avia/lib/urls';
import getPriceHintTypeForDayOfDynamicTable from 'projects/avia/components/Dynamics/utilities/getPriceHintTypeForDayOfDynamicTable';
import getNearestIndexes from 'projects/avia/lib/dynamic/getNearestIndexes';

import {
    dynamicsDashBackward,
    dynamicsDashForward,
} from 'i18n/avia-AviaDynamics';

import {
    DATES_RANGE_ADDITIONAL,
    FULL_HEIGHT_DAY,
    FULL_WIDTH_DAY,
    ROUND_TRIP_DATES_RANGE_HORIZONTAL,
    ROUND_TRIP_DATES_RANGE_VERTICAL,
} from 'projects/avia/components/Dynamics/Table/const';
import RoundButton from 'components/RoundButton/RoundButton';
import ArrowLeftIcon from 'icons/16/ArrowLeft';
import ArrowRightIcon from 'icons/16/ArrowRight';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import Flight from 'icons/16/Flight';
import ArrowTopIcon from 'icons/16/ArrowTop';
import ArrowBottomIcon from 'icons/16/ArrowBottom';
import Day from 'projects/avia/components/Dynamics/Table/Day/Day';
import DayButton from 'projects/avia/components/Dynamics/Table/DayButton/DayButton';

import cx from './SearchSuccess.scss';

export interface ISearchSuccessProps {
    requestParams: TValidDataForRequestTableDynamic;
    prices: TDynamicTableDaysInfo;
    inlineSearchRequest: TInlineSearchRequest;
    onLinkClick: (() => void) | undefined;
    datesForward: ITableDynamicCalendarDate[];
    datesBackward: ITableDynamicCalendarDate[];
    searchWhen: TDateRobot; // Дата "туда" из контекста поиска
    searchReturnDate: TDateRobot; // Дата "обратно" из контекста поиска
}

const SearchSuccess: React.FC<ISearchSuccessProps> = ({
    requestParams,
    prices,
    inlineSearchRequest,
    onLinkClick,
    datesForward,
    datesBackward,
    searchWhen,
    searchReturnDate,
}) => {
    const {searchForm, filters} = requestParams;
    const {when, return_date: returnDate} = searchForm;

    const filtersForUrl = useMemo(() => getFiltersForUrl(filters), [filters]);

    const [forwardTopIndex, setForwardTopIndex] = useState(
        Math.max(
            datesForward.findIndex(({date}) => date === when) -
                ROUND_TRIP_DATES_RANGE_VERTICAL,
            0,
        ),
    );
    const maxForwardIndex = datesForward.length - 1;
    const maxForwardTopIndex =
        maxForwardIndex - ROUND_TRIP_DATES_RANGE_VERTICAL * 2;
    const forwardBottomIndex =
        forwardTopIndex + ROUND_TRIP_DATES_RANGE_VERTICAL * 2;
    const [backwardLeftIndex, setBackwardLeftIndex] = useState(
        Math.max(
            datesBackward.findIndex(({date}) => date === returnDate) -
                ROUND_TRIP_DATES_RANGE_HORIZONTAL,
            0,
        ),
    );
    const maxBackwardIndex = datesBackward.length - 1;
    const maxBackwardLeftIndex =
        maxBackwardIndex - ROUND_TRIP_DATES_RANGE_HORIZONTAL * 2;
    const backwardRightIndex =
        backwardLeftIndex + ROUND_TRIP_DATES_RANGE_HORIZONTAL * 2;

    const onClickLeftButton = useImmutableCallback(() => {
        if (backwardLeftIndex > 0) {
            setBackwardLeftIndex(backwardLeftIndex - 1);
        }
    });
    const onClickRightButton = useImmutableCallback(() => {
        if (backwardLeftIndex < maxBackwardLeftIndex) {
            setBackwardLeftIndex(backwardLeftIndex + 1);
        }
    });
    const onClickTopButton = useImmutableCallback(() => {
        if (forwardTopIndex > 0) {
            setForwardTopIndex(forwardTopIndex - 1);
        }
    });
    const onClickBottomButton = useImmutableCallback(() => {
        if (forwardTopIndex < maxForwardTopIndex) {
            setForwardTopIndex(forwardTopIndex + 1);
        }
    });

    // Дни для рендеринга
    const forwardDaysStart = Math.max(
        forwardTopIndex - DATES_RANGE_ADDITIONAL,
        0,
    );
    const forwardDaysEnd = Math.min(
        forwardBottomIndex + DATES_RANGE_ADDITIONAL,
        maxForwardIndex,
    );
    const forwardFakeBlockHeight = forwardDaysStart * FULL_HEIGHT_DAY;
    const translateY = -forwardTopIndex * FULL_HEIGHT_DAY;
    const forwardDatesForRender = datesForward.slice(
        forwardDaysStart,
        forwardDaysEnd + 1,
    );

    const backwardDaysStart = Math.max(
        backwardLeftIndex - DATES_RANGE_ADDITIONAL,
        0,
    );
    const backwardDaysEnd = Math.min(
        backwardRightIndex + DATES_RANGE_ADDITIONAL,
        maxBackwardIndex,
    );
    const backwardFakeBlockWidth = backwardDaysStart * FULL_WIDTH_DAY;
    const translateX = -backwardLeftIndex * FULL_WIDTH_DAY;
    const backwardDatesForRender = datesBackward.slice(
        backwardDaysStart,
        backwardDaysEnd + 1,
    );

    // Формируем данные дней для рендера
    const daysForRender: IDynamicsTableCalendarDay[][] =
        forwardDatesForRender.map(dateForward =>
            backwardDatesForRender.map(dateBackward => {
                return {
                    ...getDynamicDayForTable(
                        {dateForward, dateBackward},
                        prices,
                    ),
                    type: getPriceHintTypeForDayOfDynamicTable({
                        prices,
                        datesForward,
                        datesBackward,
                        dateForward,
                        dateBackward,
                    }),
                    url: aviaURLs.getSearchResultsUrl(
                        {
                            ...searchForm,
                            when: dateForward.date,
                            return_date: dateBackward.date,
                            linkSource: EAviaLinkSource.DYNAMIC,
                        },
                        filtersForUrl,
                    ),
                };
            }),
        );

    // Запускаем inline-поиски для дней без цен или с примерными ценами
    useEffect(() => {
        const forwardDayIndexes = getNearestIndexes(
            datesForward,
            datesForward.findIndex(({date}) => date === when),
            1,
        );
        const backwardDayIndexes = getNearestIndexes(
            datesBackward,
            datesBackward.findIndex(({date}) => date === returnDate),
            1,
        );

        const datesForRequest: {
            forwardDate: TDateRobot;
            backwardDate: TDateRobot;
        }[] = [];

        for (
            let forwardIndex = forwardDayIndexes.start;
            forwardIndex <= forwardDayIndexes.end;
            forwardIndex++
        ) {
            const dateForward = datesForward[forwardIndex];

            for (
                let backwardIndex = backwardDayIndexes.start;
                backwardIndex <= backwardDayIndexes.end;
                backwardIndex++
            ) {
                const dateBackward = datesBackward[backwardIndex];

                // Не запрашиваем текущий контекст поиска
                if (
                    dateForward.date === when &&
                    dateBackward.date === returnDate
                ) {
                    continue;
                }

                // Исключаем невозможные даты ("обратно" раньше "туда")
                if (dateBackward.date < dateForward.date) {
                    continue;
                }

                const {status} = getDynamicDayForTable(
                    {dateForward, dateBackward},
                    prices,
                );

                if (status === EDynamicsDayStatus.SHOULD_SEARCH) {
                    datesForRequest.push({
                        forwardDate: dateForward.date,
                        backwardDate: dateBackward.date,
                    });
                }
            }
        }

        // Запускаем инлайн-поиски
        batch(() => {
            datesForRequest.forEach(({forwardDate, backwardDate}) => {
                inlineSearchRequest(forwardDate, backwardDate);
            });
        });
    }, [
        datesBackward,
        datesForward,
        inlineSearchRequest,
        prices,
        returnDate,
        when,
    ]);

    return (
        <div className={cx('root')}>
            <div className={cx('backwardButtons')}>
                <TextWithIcon
                    text={dynamicsDashBackward()}
                    iconLeft={Flight}
                    iconLeftClassName={cx('backwardFlight')}
                />
                <RoundButton
                    className={cx('leftButton')}
                    size="s"
                    theme="white"
                    icon={<ArrowLeftIcon />}
                    disabled={backwardLeftIndex === 0}
                    onClick={onClickLeftButton}
                />
                <RoundButton
                    className={cx('rightButton')}
                    size="s"
                    theme="white"
                    icon={<ArrowRightIcon />}
                    disabled={backwardLeftIndex === maxBackwardLeftIndex}
                    onClick={onClickRightButton}
                />
            </div>

            <div className={cx('backwardContainer')}>
                <div className={cx('backwardDatesViewport')}>
                    <div
                        className={cx('backwardDatesContainer')}
                        style={{transform: `translateX(${translateX}px)`}}
                    >
                        <div
                            className={cx('fakeBackwardBlock')}
                            style={{width: backwardFakeBlockWidth}}
                        />

                        {backwardDatesForRender.map(({date, isHoliday}) => (
                            <Day
                                className={cx('backwardDate')}
                                key={date}
                                date={date}
                                isHoliday={isHoliday}
                            />
                        ))}
                    </div>
                </div>
            </div>

            <div className={cx('mainContainer')}>
                <div className={cx('forwardButtons')}>
                    <RoundButton
                        className={cx('topButton')}
                        size="s"
                        theme="white"
                        icon={<ArrowTopIcon />}
                        disabled={forwardTopIndex === 0}
                        onClick={onClickTopButton}
                    />
                    <RoundButton
                        className={cx('bottomButton')}
                        size="s"
                        theme="white"
                        icon={<ArrowBottomIcon />}
                        disabled={forwardTopIndex === maxForwardTopIndex}
                        onClick={onClickBottomButton}
                    />
                    <span className={cx('backwardText')}>
                        {dynamicsDashForward()}
                    </span>
                    <Flight className={cx('forwardFlight')} />
                </div>

                <div className={cx('forwardDatesViewport')}>
                    <div
                        className={cx('forwardDatesContainer')}
                        style={{transform: `translateY(${translateY}px)`}}
                    >
                        <div
                            className={cx('fakeForwardBlock')}
                            style={{height: forwardFakeBlockHeight}}
                        />
                        {forwardDatesForRender.map(({date, isHoliday}) => (
                            <Day
                                className={cx('forwardDate')}
                                key={date}
                                date={date}
                                isHoliday={isHoliday}
                            />
                        ))}
                    </div>
                </div>

                <div className={cx('datesViewPort')}>
                    <div
                        className={cx('datesVerticalContainer')}
                        style={{
                            transform: `translateX(${translateX}px) translateY(${translateY}px)`,
                        }}
                    >
                        <div
                            className={cx('fakeForwardBlock')}
                            style={{height: forwardFakeBlockHeight}}
                        />

                        <div className={cx('datesHorizontalContainer')}>
                            <div
                                className={cx('fakeBackwardBlock')}
                                style={{width: backwardFakeBlockWidth}}
                            />

                            <div className={cx('datesContainer')}>
                                {daysForRender.map(horizontalDates => (
                                    <div
                                        className={cx('horizontalDates')}
                                        key={
                                            horizontalDates[0].dateForward.date
                                        }
                                    >
                                        {horizontalDates.map(day => (
                                            <DayButton
                                                className={cx('day')}
                                                key={day.dateBackward.date}
                                                day={day}
                                                isCurrent={
                                                    day.dateForward.date ===
                                                        searchWhen &&
                                                    day.dateBackward.date ===
                                                        searchReturnDate
                                                }
                                                inlineSearchRequest={
                                                    inlineSearchRequest
                                                }
                                                onLinkClick={onLinkClick}
                                            />
                                        ))}
                                    </div>
                                ))}
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default React.memo(SearchSuccess);
