import moment from 'moment';

import {IDynamicsChartData} from 'projects/avia/components/Dynamics/types/IDynamicsChartData';
import {
    EDynamicsDayStatus,
    IDynamicsDay,
} from 'projects/avia/components/Dynamics/types/IDynamicsDay';

import {
    TDynamicsDaysInfo,
    createFakePrice,
} from 'reducers/avia/aviaPriceIndex/utils/convertPriceIndexDataToDynamicsData';

import {PriceComparator} from 'utilities/currency/compare';
import getDateRobotFromMoment from 'utilities/dateUtils/getDateRobotFromMoment';
import {isHoliday} from 'utilities/dateUtils/isHoliday';

export function dynamicsPriceDays(
    daysInfo: TDynamicsDaysInfo,
    delta: number | null,
    priceComparator: PriceComparator,
): IDynamicsChartData {
    const startDate = moment.utc();
    const endDate = startDate.clone().add(1, 'year').subtract(1, 'day');

    return getDynamicsPriceDays(
        startDate,
        endDate,
        daysInfo,
        delta,
        priceComparator,
    );
}

export function getDynamicsPriceDays(
    startDate: moment.Moment,
    endDate: moment.Moment,
    daysInfo: TDynamicsDaysInfo,
    delta: number | null,
    priceComparator: PriceComparator,
): IDynamicsChartData {
    const days: IDynamicsDay[] = [];
    let minPrice: IDynamicsChartData['minPrice'];
    let maxPrice: IDynamicsChartData['maxPrice'];

    while (startDate.isSameOrBefore(endDate)) {
        const dateForward = getDateRobotFromMoment(startDate);

        const info = daysInfo[dateForward]
            ? daysInfo[dateForward]
            : createFakePrice();

        if (info.status === EDynamicsDayStatus.HAS_DATA && info.price) {
            minPrice = priceComparator.min(info.price, minPrice);
            maxPrice = priceComparator.max(info.price, maxPrice);
        }

        const day: IDynamicsDay = {
            dateForward: {
                date: dateForward,
                isHoliday: isHoliday(startDate),
            },
            dateBackward: null,
            progress: info.progress,
            status: info.status,
            price: info.price,
        };

        if (delta !== null) {
            const backwardMoment = startDate.clone().add(delta, 'days');
            const dateBackward = getDateRobotFromMoment(backwardMoment);

            day.dateBackward = {
                date: dateBackward,
                isHoliday: isHoliday(backwardMoment),
            };
        }

        days.push(day);
        startDate.add(1, 'day');
    }

    return {minPrice, maxPrice, days};
}
