import {minBy, maxBy} from 'lodash';

import {IAviaPriceIndexPrice} from 'server/api/AviaPriceIndexApi/types/IAviaPriceIndexPrice';

import {RANGE_DATES} from 'projects/avia/components/Dynamics/utilities/dynamicsWeekDates';
import {
    EPriceHintType,
    getPriceHintType,
} from 'projects/avia/components/Dynamics/utilities/getPriceHintType';
import getNearestIndexes from 'projects/avia/lib/dynamic/getNearestIndexes';

const CALENDAR_USD_LIMIT = 10;

export interface IPriceDay {
    price: IAviaPriceIndexPrice | null;
}

/**
 * Возвращает тип цены для дня, относительно массива дней
 *
 * Информация о цене из day должна так же присутствовать в days.
 */
export function getDynamicsCalendarDayType(
    day: IPriceDay,
    days: IPriceDay[],
): EPriceHintType {
    if (!day.price || !days.length) {
        return EPriceHintType.EMPTY;
    }

    const prices = days.map(item => item.price).filter(Boolean);
    const allWithPrices = days.length === prices.length;
    const minPrice = minBy(prices, 'value');
    const maxPrice = maxBy(prices, 'value');

    if (
        !minPrice ||
        minPrice.value === maxPrice?.value ||
        !allWithPrices ||
        minPrice.roughly ||
        day.price.roughly
    ) {
        return EPriceHintType.EMPTY;
    }

    return minPrice.value === day.price.value
        ? EPriceHintType.MINIMAL
        : getPriceHintType({
              currentPrice: day.price,
              weekMinPrice: minPrice,
              allWithPrices,
              limit: CALENDAR_USD_LIMIT,
          });
}

/**
 * Считает тип цены конкретного дня, относительно фрагмента из RANGE_DATES*2,
 * ближайших дней
 * @param days
 * @param indexDay - индекс дня, для которого нужно выполнить расчет
 */
export function getWeekDynamicsCalendarDayType(
    days: IPriceDay[],
    indexDay: number,
): EPriceHintType {
    // Берем фрагмент из RANGE_DATES*2, ближайших к переданному дню
    const {start, end} = getNearestIndexes(days, indexDay, RANGE_DATES);

    return getDynamicsCalendarDayType(
        days[indexDay],
        days.slice(start, end + 1),
    );
}
