import {IAviaPriceIndexPrice} from 'server/api/AviaPriceIndexApi/types/IAviaPriceIndexPrice';
import TDateRobot from 'types/common/date/TDateRobot';

import IPrice from 'utilities/currency/PriceInterface';
import {PriceComparator} from 'utilities/currency/compare';
import {
    EPriceHintType,
    getPriceHintType,
} from 'projects/avia/components/Dynamics/utilities/getPriceHintType';

import {sortDynamicsPrices} from './sortDynamicsPrices';

export interface IPriceHint {
    type: EPriceHintType;
    otherPrice?: IPrice;
    minPrice?: IPrice;
}

export interface IDayWithPrice {
    date: TDateRobot;
    price?: IAviaPriceIndexPrice;
}

/**
 * Получает массив цен и минимальную цену, возвращает тип минимальной цены относительно цен в массиве
 * и 2 минимальные цены из этого массива
 * , список цен из price index и цену поиска.
 * Возвращает тип цены поиска относительно цен из price index в периоде переданных дат.
 * Дополняет логику getPriceHintType - возвращает EPriceHintType.EMPTY для случаев, когда недостаточно данных для
 * определения типа в getPriceHintType
 */
export function getPriceHintInfo(
    daysWithPrice: IDayWithPrice[],
    priceComparator: PriceComparator,
    minSearchPrice?: Nullable<IPrice>,
): IPriceHint {
    if (!daysWithPrice || !daysWithPrice.length || !minSearchPrice) {
        return {
            type: EPriceHintType.EMPTY,
        };
    }

    const {
        0: minPriceDay,
        1: otherPriceDay,
        [daysWithPrice.length - 1]: maybeWithoutPriceDay,
    } = [...daysWithPrice].sort((a, b) =>
        sortDynamicsPrices(a, b, priceComparator),
    );
    // Если цена на текущую дату и минимальная цена на неделю совпадает, то для сравнения
    // используем следующую минимальную цену, чтобы понимать является ли цена минимальной или приемлемой
    const weekMinPrice =
        minPriceDay.price &&
        priceComparator.isEqualPrice(minSearchPrice, minPriceDay.price)
            ? otherPriceDay.price
            : minPriceDay.price;

    if (!weekMinPrice || weekMinPrice.roughly) {
        return {
            type: EPriceHintType.EMPTY,
        };
    }

    // Считается, что цены есть на каждый день, если они все точные (!roughly)
    const allWithPrices = Boolean(
        maybeWithoutPriceDay.price && !maybeWithoutPriceDay.price.roughly,
    );

    const type = getPriceHintType({
        currentPrice: minSearchPrice,
        weekMinPrice,
        allWithPrices,
    });

    return {
        type,
        otherPrice: otherPriceDay.price,
        minPrice: minPriceDay.price,
    };
}
