import React, {Children, useContext} from 'react';

import {IWithClassName} from 'types/withClassName';

import cx from './FlightTariffTable.scss';

interface FlightTariffTableProps {
    activeColumn?: number;
    children?: React.ReactNode;
    width?: 'max' | 'auto';
}

const TableContext = React.createContext<number | null | undefined>(undefined);

export function FlightTariffTable({
    children,
    activeColumn,
    width,
}: FlightTariffTableProps) {
    return (
        <TableContext.Provider value={activeColumn}>
            <table className={cx('table', width && `table_width_${width}`)}>
                <tbody>{children}</tbody>
            </table>
        </TableContext.Provider>
    );
}

interface FlightTariffTableRowProps {
    borders?: {top?: boolean; bottom?: boolean};
    children?: React.ReactNode;
}

const ActiveCellContext = React.createContext(false);

FlightTariffTable.Row = function FlightTariffTableRow({
    borders = {},
    children,
}: FlightTariffTableRowProps) {
    const activeColumn = useContext(TableContext);
    const {bottom: bottomBorder, top: topBorder} = borders;

    return (
        <tr
            className={cx(
                'row',
                bottomBorder && 'row_border_bottom',
                topBorder && 'row_border_top',
            )}
        >
            {Children.map(children, (child, index) => (
                <ActiveCellContext.Provider value={index === activeColumn}>
                    {child}
                </ActiveCellContext.Provider>
            ))}
        </tr>
    );
};

interface FlightTariffTableCellProps extends IWithClassName {
    children?: React.ReactNode;
}

FlightTariffTable.Cell = function FlightTariffTableCell({
    children,
    className,
}: FlightTariffTableCellProps) {
    const isActive = useContext(ActiveCellContext);

    return (
        <td className={cx('cell', isActive && 'cell_active', className)}>
            {children}
        </td>
    );
};
