import React, {useReducer, useEffect, useRef} from 'react';
import {useInView} from 'react-intersection-observer';
import {createAction, createReducer, ActionType} from 'typesafe-actions';
import {noop} from 'lodash';

import useImmutableCallback from 'utilities/hooks/useImmutableCallback';

import Intersperse from 'components/Intersperse/Intersperse';

interface ILazyListActionPayload {
    slice: React.ReactNode;
    shownItems: any[];
}

interface ILazyListResetActionPayload extends ILazyListActionPayload {
    items: any[];
}

const actions = {
    addSlice: createAction('ADD_SLICE')<ILazyListActionPayload>(),
    reset: createAction('RESET')<ILazyListResetActionPayload>(),
};

interface ILazyListState {
    page: number;
    items: any[];
    shownItems: any[];
    slices: React.ReactNode[];
}

const reducer = createReducer<ILazyListState, ActionType<typeof actions>>({
    page: 0,
    items: [],
    shownItems: [],
    slices: [],
})
    .handleAction(
        actions.addSlice,
        ({items, page, slices, shownItems}, {payload}) => ({
            items,
            page: page + 1,
            shownItems: shownItems.concat(payload.shownItems),
            slices: slices.concat([payload.slice]),
        }),
    )
    .handleAction(actions.reset, (state, {payload}) => ({
        page: 1,
        items: payload.items,
        shownItems:
            state.items === payload.items ? state.shownItems : payload.items,
        slices: [payload.slice],
    }));

const INTERSECTION_OBSERVER_PROPS = {
    rootMargin: '0px 0px 500px 0px',
};

interface IAviaVariantsLazyListProps<T> {
    items: T[];
    renderSeparator: (idx: number) => React.ReactNode;
    renderSlice(items: T[], idx: number): React.ReactNode;
    sliceSize?: number;
    onScroll?: (page: number) => void;
    onRender?: (data: T[], count: number, page: number) => void;
}

export function LazyList<T>({
    items,
    renderSlice,
    renderSeparator,
    sliceSize = 10,
    onRender = noop,
    onScroll = noop,
}: IAviaVariantsLazyListProps<T>) {
    const isMountedRef = useRef(false);
    const initialData = items.slice(0, sliceSize);
    const initState: ILazyListState = {
        page: 1,
        items,
        shownItems: initialData,
        slices: [renderSlice(initialData, 0)],
    };
    const [{page, slices, shownItems}, dispatch] = useReducer(
        reducer,
        initState,
    );
    const totalPages = Math.ceil(items.length / sliceSize);

    const [ref, hasIntersected] = useInView(INTERSECTION_OBSERVER_PROPS);
    const addSlice = useImmutableCallback(() => {
        if (page >= totalPages) {
            return;
        }

        const start = page * sliceSize;
        const end = start + sliceSize;
        const itemsToRender = items.slice(start, end);

        dispatch(
            actions.addSlice({
                shownItems: itemsToRender,
                slice: renderSlice(itemsToRender, page),
            }),
        );

        onScroll(page + 1);
    });

    // Хук должен отслеживать только изменения пропсов, на маунт мы его пропускаем
    useEffect(() => {
        if (isMountedRef.current) {
            const itemsToRender = items.slice(0, sliceSize);

            dispatch(
                actions.reset({
                    items: items,
                    shownItems: itemsToRender,
                    slice: renderSlice(itemsToRender, 0),
                }),
            );
        } else {
            isMountedRef.current = true;
        }
    }, [dispatch, items, sliceSize, renderSlice, isMountedRef]);

    // Эффект зависит от внутреннего состояния компонента а не от внешних items,
    // в противном случае, если мы будем отслеживать и внешнее состояние (items)
    // и внутреннее (page), то это может привести к лишним вызовам хука.
    useEffect(() => {
        const start = sliceSize * (page - 1);
        const end = start + sliceSize;

        onRender(shownItems.slice(start, end), start, page);
    }, [onRender, shownItems, page, sliceSize]);

    useEffect(() => {
        if (hasIntersected) {
            addSlice();
        }
    }, [hasIntersected, addSlice]);

    return (
        <>
            <Intersperse separator={renderSeparator}>{slices}</Intersperse>
            {page < totalPages && <div ref={ref} />}
        </>
    );
}
