import React, {Component, ReactNode} from 'react';
import {IYMapsApi} from 'react-yandex-maps';

import {IPrepareNearAnswer} from './types';
import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';

import {prepareNearAnswer} from 'projects/avia/lib/nearAirports/prepareNearAnswer';

import Spinner from 'components/Spinner/Spinner';
import YandexMapsComponent from 'components/YandexMaps/YandexMaps';
import NearAirportsList from 'projects/avia/components/NearAirportsList/NearAirportsList';
import AviaMapMarker from 'projects/avia/components/AviaMapMarker/AviaMapMarker';
import {Center} from 'projects/avia/components/Center/Center';

import aviaBrowserProvider from 'serviceProvider/avia/aviaBrowserProvider';

import cx from './MapNearAirports.scss';

const DEFAULT_ZOOM = 6;

export enum ENearAirportsMapState {
    FETCHING,
    ERROR,
    HAS_POINTS,
    NO_POINTS,
}

export interface IMapNearAirportsState {
    mapInfo?: IPrepareNearAnswer;
    hasError: boolean;
    mapSuccessLoaded: boolean;
}

interface IMapNearAirportsProps {
    formParams: IAviaParams;
    onStateChange?: (state: ENearAirportsMapState) => void;
    view?: 'modal' | 'yak';
}

class MapNearAirports extends Component<
    IMapNearAirportsProps,
    IMapNearAirportsState
> {
    private yandexMapAPIInstance: IYMapsApi | null = null;

    static defaultProps = {
        view: 'modal' as IMapNearAirportsProps['view'],
    };

    state: IMapNearAirportsState = {
        hasError: false,
        mapSuccessLoaded: false,
    };

    componentDidMount(): void {
        const {formParams} = this.props;

        aviaBrowserProvider
            .getNearAirports(formParams)
            .then(nearAnswer => {
                const preparedNearAnswer = prepareNearAnswer(
                    nearAnswer,
                    location.href,
                );

                this.setStateNotified({mapInfo: preparedNearAnswer});
            })
            .catch(() => {
                this.setStateNotified({hasError: true});
            });
    }

    private renderAviaMarkers(): React.ReactNode {
        const {mapInfo, mapSuccessLoaded} = this.state;

        if (!mapInfo || !mapSuccessLoaded) {
            return null;
        }

        const markers = mapInfo.nearPoints.map(nearPoint => {
            return (
                <AviaMapMarker
                    key={`${nearPoint.point[0]}-${nearPoint.point[1]}`}
                    item={nearPoint}
                    yandexMapAPIInstance={this.yandexMapAPIInstance}
                />
            );
        });

        markers.push(
            <AviaMapMarker
                key={`${mapInfo.toPoint.point[0]}-${mapInfo.toPoint.point[1]}`}
                item={mapInfo.toPoint}
                center={true}
                yandexMapAPIInstance={this.yandexMapAPIInstance}
            />,
        );

        return markers;
    }

    private handleLoadMap = (yandexMapAPIInstance: IYMapsApi): void => {
        if (yandexMapAPIInstance) {
            this.yandexMapAPIInstance = yandexMapAPIInstance;

            this.setStateNotified({mapSuccessLoaded: true});
        }
    };

    private setStateNotified(state: Partial<IMapNearAirportsState>): void {
        this.setState(state as any, () => {
            if (this.props.onStateChange) {
                this.props.onStateChange(this.getNearAirportsMapState());
            }
        });
    }

    private getNearAirportsMapState(): ENearAirportsMapState {
        if (this.state.hasError) {
            return ENearAirportsMapState.ERROR;
        }

        if (!this.state.mapInfo || !this.state.mapSuccessLoaded) {
            return ENearAirportsMapState.FETCHING;
        }

        if (this.state.mapInfo.nearPoints.length) {
            return ENearAirportsMapState.HAS_POINTS;
        }

        return ENearAirportsMapState.NO_POINTS;
    }

    render(): ReactNode {
        const {hasError, mapInfo} = this.state;

        if (hasError) {
            // TODO не достучались до апи
            return (
                <Center>
                    Произошла ошибка. Попробуйте перезагрузить страницу
                </Center>
            );
        }

        if (!mapInfo) {
            return (
                <Center>
                    <Spinner />
                </Center>
            );
        }

        return (
            <YandexMapsComponent
                defaultZoom={DEFAULT_ZOOM}
                hasZoomControl={false}
                defaultCenter={{
                    lat: mapInfo.toPoint.point[0],
                    lon: mapInfo.toPoint.point[1],
                }}
                onLoadMap={this.handleLoadMap}
                className={cx('map', `_view_${this.props.view}`)}
            >
                {this.renderAviaMarkers()}
                <NearAirportsList
                    mapInfo={mapInfo}
                    className={cx('nearAirportsList')}
                />
            </YandexMapsComponent>
        );
    }
}

export default MapNearAirports;
