import {FunctionComponent} from 'react';

import {IWithClassName} from 'types/withClassName';
import {
    IPrepareNearAnswer,
    IPrepareNearItem,
} from 'projects/avia/components/MapNearAirports/types';

import * as i18nBlock from 'i18n/avia-NearAirports';

import ButtonLink from 'components/ButtonLink/ButtonLink';
import Price from 'components/Price/Price';
import Text from 'components/Text/Text';

import cx from './NearAirportsList.scss';

export interface INearAirportsProps {
    nearPoint: IPrepareNearItem;
    targetCityPhrase: string;
    showTargetCity: boolean;
}

export interface INearAirportsListProps extends IWithClassName {
    mapInfo: IPrepareNearAnswer;
}

const NearAirport: FunctionComponent<INearAirportsProps> = ({
    nearPoint,
    showTargetCity,
    targetCityPhrase,
}) => {
    const distance = i18nBlock.km({
        distance: Math.round(nearPoint.distance),
        city: showTargetCity && targetCityPhrase,
    });

    return (
        <div className={cx('nearPoint')}>
            <div className={cx('nearPoint__leftColumn')}>
                <div className={cx('nearPoint__location')}>
                    <span className={cx('location__fullTitle')}>
                        {nearPoint.title}
                    </span>
                    <span>, {nearPoint.country}</span>
                </div>
                <div className={cx('nearPoint__distance')}>{distance}</div>
            </div>
            <ButtonLink
                theme="secondary"
                size="m-inset"
                className={cx('nearPoint__buyButton')}
                to={nearPoint.url}
            >
                {nearPoint.price ? (
                    <Price {...nearPoint.price} isFrom />
                ) : (
                    i18nBlock.buttonDashFind()
                )}
            </ButtonLink>
        </div>
    );
};

const NearAirportsList: FunctionComponent<INearAirportsListProps> = ({
    mapInfo: {nearPoints, toPoint},
    className,
}) => {
    return (
        <div
            className={cx(
                'root',
                {
                    root_empty: nearPoints.length === 0,
                },
                className,
            )}
        >
            {nearPoints.length === 0 && (
                <Text color="primary" size="m">
                    {i18nBlock.noAirports()}
                </Text>
            )}
            {nearPoints.length > 0 &&
                nearPoints.map((nearPoint, i) => {
                    return (
                        <NearAirport
                            key={nearPoint.id}
                            nearPoint={nearPoint}
                            showTargetCity={i === 0}
                            targetCityPhrase={toPoint.phraseFrom}
                        />
                    );
                })}
        </div>
    );
};

export default NearAirportsList;
