import React, {FunctionComponent, useRef} from 'react';

import {IWithClassName} from 'src/types/withClassName';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import useTooltipHover from 'utilities/hooks/useTooltipHover';
import {EAviaOfferType} from 'projects/avia/lib/search/offerType';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useBoolean} from 'utilities/hooks/useBoolean';

import * as i18nBlock from 'i18n/avia-charter-hint';
import * as i18nAviaDisclaimerBlock from 'i18n/avia_disclaimer';

import BottomSheet from 'components/BottomSheet/BottomSheet';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';
import LinkButton from 'components/LinkButton/LinkButton';
import Box from 'components/Box/Box';
import InfoIcon from 'icons/16/Info';

import cx from './OfferTypeHint.scss';

interface IOfferTypeHintDeviceProps extends IWithClassName {
    trigger: React.ReactElement;
    description: React.ReactElement;
}

export interface IOfferTypeHintProps extends IWithClassName {
    type: EAviaOfferType;
}

const OfferTypeHintMobile: FunctionComponent<IOfferTypeHintDeviceProps> = ({
    trigger,
    description,
}) => {
    const {value: isVisible, setTrue: show, setFalse: hide} = useBoolean(false);
    const triggerRef = useRef(null);

    return (
        <>
            <LinkButton
                theme="ghost"
                className={cx('hintButton')}
                onClick={show}
                innerRef={triggerRef}
            >
                {trigger}
            </LinkButton>

            <BottomSheet isOpened={isVisible} onClose={hide}>
                {description}
            </BottomSheet>
        </>
    );
};

const DIRECTION_DEFAULT = [
    EPopupDirection.BOTTOM_RIGHT,
    EPopupDirection.TOP_RIGHT,
];

const OfferTypeHintDesktop: FunctionComponent<IOfferTypeHintDeviceProps> = ({
    trigger,
    description,
}) => {
    const {isHovered, handleMouseEnter, handleMouseLeave} = useTooltipHover();
    const triggerRef = useRef(null);

    return (
        <>
            <LinkButton
                theme="ghost"
                className={cx('hintButton')}
                onMouseEnter={handleMouseEnter}
                onMouseLeave={handleMouseLeave}
                innerRef={triggerRef}
            >
                {trigger}
            </LinkButton>

            <MessageBoxPopup
                isVisible={isHovered}
                anchorRef={triggerRef}
                direction={DIRECTION_DEFAULT}
                onClose={handleMouseLeave}
            >
                {description}
            </MessageBoxPopup>
        </>
    );
};

const OfferTypeHint: FunctionComponent<IOfferTypeHintProps> = ({type}) => {
    const deviceType = useDeviceType();
    const {isMobile, isDesktop} = deviceType;
    const charter = type === EAviaOfferType.charter;

    const shortMessage = charter
        ? i18nBlock.charter()
        : i18nBlock.specialDashConditions();

    const trigger = (
        <TextWithIcon
            size="s"
            spaceBetween={2}
            text={shortMessage}
            iconRight={InfoIcon}
        />
    );

    const description = (
        <Box
            className={cx(
                'hintDescription',
                deviceMods('hintDescription', deviceType),
            )}
            between="5"
        >
            <Heading className={cx('heading')} level={isMobile ? 2 : 3}>
                {shortMessage}
            </Heading>
            <Text size="s" color={isMobile ? 'primary' : 'inverse'}>
                {charter
                    ? i18nAviaDisclaimerBlock.aviaCharterAboutOrderContent()
                    : i18nAviaDisclaimerBlock.aviaSpecialConditionsContent()}
            </Text>
        </Box>
    );

    return (
        <>
            {isDesktop && (
                <OfferTypeHintDesktop
                    trigger={trigger}
                    description={description}
                />
            )}
            {isMobile && (
                <OfferTypeHintMobile
                    trigger={trigger}
                    description={description}
                />
            )}
        </>
    );
};

export default OfferTypeHint;
