/* eslint-disable camelcase */
import React, {useCallback} from 'react';
import {parse, stringify} from 'query-string';
import {Location} from 'history';
import {useSelector} from 'react-redux';

import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {IWithClassName} from 'types/withClassName';
import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {filtersHashSelector} from 'selectors/avia/search/filters/filtersHashSelector';

import {reachGoal} from 'utilities/metrika';
import {aviaURLs} from 'projects/avia/lib/urls';
import {restoreYtpReferer} from 'utilities/ytpReferer';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Link from 'components/Link/Link';

import cx from './OrderLink.scss';

interface IOrderLinkProps extends IWithClassName, IWithQaAttributes {
    variant: IResultAviaVariant;
    withBaggage?: boolean;
    showFocuseOutline?: boolean;
    style?: React.CSSProperties;
}

const OrderLink: React.FC<IOrderLinkProps> = props => {
    const {
        className,
        variant,
        showFocuseOutline = false,
        style,
        children,
    } = props;

    const onClick = useCallback(() => {
        reachGoal(EAviaGoal.SEARCH_REDIRECT);
    }, []);

    const filtersHash = useSelector(filtersHashSelector);

    const getOrderLocationDescriptor = useCallback(
        (location: Location) => {
            const {
                backwardRoute: backward,
                forwardRoute: forward,
                key,
            } = variant;

            const {ytp_referer = restoreYtpReferer(), ...other} = parse(
                location.search,
            );

            const searchParams: Partial<Record<keyof IAviaParams, string>> = {
                ...other,
                forward,
                backward,
                baggage: variant.price.hasBaggage ? '1' : undefined,
                free_refund: variant.price.hasFreeRefund ? '1' : undefined,
                charge_refund: variant.price.hasChargedRefund ? '1' : undefined,
                ytp_referer:
                    typeof ytp_referer === 'string'
                        ? ytp_referer
                        : ytp_referer?.toString(),
            };

            return {
                pathname: aviaURLs.getOrderUrl(),
                search: stringify(searchParams),
                hash: filtersHash,
                state: {
                    variantKey: key,
                    background: location,
                    scrollTop: null,
                },
            };
        },
        [filtersHash, variant],
    );

    return (
        <Link
            className={cx(
                showFocuseOutline ? null : 'orderLinkHideOutline',
                className,
            )}
            to={getOrderLocationDescriptor}
            onClick={onClick}
            style={style}
            {...prepareQaAttributes(props)}
        >
            {children}
        </Link>
    );
};

OrderLink.displayName = 'OrderLink';

export default OrderLink;
