import React, {useEffect, useState} from 'react';

import {IPassengerExperience} from 'projects/avia/types/IPassengerExperience';
import {SeatsInRowType} from 'projects/avia/types/SeatsInRowType';

import * as localized from 'i18n/avia-PassengerExperience';

import {aviaPassengerExperienceProvider} from 'serviceProvider/avia/aviaPassengerExperienceProvider';

interface IPassengerExperienceRenderProps {
    passengerExperience: IPassengerExperience | null;
    isLoading: boolean;
}

interface IPassengerExperienceProps {
    when: string;
    klass?: string;
    iata: string;
    flightNumber: string;
    children: (props: IPassengerExperienceRenderProps) => React.ReactNode;
}

interface IPassengerExperienceState {
    passengerExperience: IPassengerExperience | null;
    isLoading: boolean;
}

export const PassengerExperience: React.FC<IPassengerExperienceProps> =
    props => {
        const [state, setState] = useState<IPassengerExperienceState>({
            passengerExperience: null,
            isLoading: false,
        });

        useEffect(() => {
            let ignore = false;

            async function fetchData(): Promise<void> {
                setState({passengerExperience: null, isLoading: true});

                let passengerExperience;

                try {
                    passengerExperience = await aviaPassengerExperienceProvider
                        .provider()
                        .getPassengerExperience({
                            planeNumber: `${props.iata} ${Number(
                                props.flightNumber,
                            )}`,
                            klass:
                                props.klass === 'business'
                                    ? 'business'
                                    : 'economy',
                            when: props.when,
                        });
                } catch (ex) {
                    passengerExperience = null;
                }

                if (!ignore) {
                    setState({passengerExperience, isLoading: false});
                }
            }

            fetchData();

            return (): void => {
                ignore = true;
            };
        }, [props.flightNumber, props.iata, props.klass, props.when]);

        return props.children(state) as React.ReactElement<any>;
    };

export function localizeSeatsInRow(seatsInRow: SeatsInRowType): string {
    switch (seatsInRow) {
        case 'standard':
            return localized.seatsDashInDashRowDashStandard();
        case 'tight':
            return localized.seatsDashInDashRowDashTight();
        case 'wire':
            return localized.seatsDashInDashRowDashWire();
    }
}

export function localizeSeatsTotal(seatsTotal: number): string {
    return localized.seatsDashTotal({count: seatsTotal});
}

export function localizeSeatPeach(peach: number): string {
    return localized.seatsDashPeach({peach});
}
