import React from 'react';

import {IPassengerExperience} from 'projects/avia/types/IPassengerExperience';

import {CHAR_LIST_MARKER} from 'utilities/strings/charCodes';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as keyset from 'i18n/avia-PassengerExperience';

import {Icon} from 'components/Icon/Icon';
import WiFiIcon from 'icons/16/WiFi';
import SocketIcon from 'icons/16/Socket';
import EntertainmentIcon from 'icons/16/Entertainment';

import {
    localizeSeatPeach,
    localizeSeatsInRow,
    localizeSeatsTotal,
} from './PassengerExperience';

interface IPassengerExperienceOptionProps extends IWithQaAttributes {
    optionClassName?: string;
    iconClassName?: string;
    labelClassName?: string;
    icon?: React.ReactNode;
    label: React.ReactNode;
}

const Option: React.FC<IPassengerExperienceOptionProps> = props => (
    <div className={props.optionClassName} {...prepareQaAttributes(props)}>
        {props.icon && (
            <span className={props.iconClassName}>{props.icon}</span>
        )}{' '}
        <span className={props.labelClassName}>{props.label}</span>
    </div>
);

interface IPassengerExperienceItemProps extends IWithQaAttributes {
    field:
        | 'plane-model'
        | 'seat-pitch'
        | 'seats-total'
        | 'wifi'
        | 'power'
        | 'ife';
    data: IPassengerExperience;
    optionClassName?: string;
    iconClassName?: string;
    labelClassName?: string;
    children?: never;
}

export const PassengerExperienceItem: React.FC<IPassengerExperienceItemProps> =
    props => {
        const {field, data, iconClassName, labelClassName, optionClassName} =
            props;

        switch (field) {
            case 'plane-model':
                // eslint-disable-next-line no-eq-null
                if (data.planeModel == null) {
                    return null;
                }

                return (
                    <Option
                        iconClassName={iconClassName}
                        labelClassName={labelClassName}
                        optionClassName={optionClassName}
                        label={data.planeModel}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'planeModel',
                        })}
                    />
                );
            case 'seat-pitch': {
                // eslint-disable-next-line no-eq-null
                if (data.seatPitch == null) {
                    return null;
                }

                return (
                    <Option
                        iconClassName={iconClassName}
                        labelClassName={labelClassName}
                        optionClassName={optionClassName}
                        label={localizeSeatPeach(data.seatPitch)}
                    />
                );
            }
            case 'seats-total': {
                const seatsTotal = data.seatsTotal;
                const seatsInRow = data.seatsInRow;
                const localizedSeatsInRow =
                    seatsInRow && localizeSeatsInRow(seatsInRow);
                const localizedSeatsTotal =
                    seatsTotal && localizeSeatsTotal(seatsTotal);

                return (
                    <Option
                        iconClassName={iconClassName}
                        labelClassName={labelClassName}
                        optionClassName={optionClassName}
                        label={[localizedSeatsInRow, localizedSeatsTotal]
                            .filter(Boolean)
                            .join(` ${CHAR_LIST_MARKER} `)}
                    />
                );
            }
            case 'wifi': {
                // eslint-disable-next-line no-eq-null
                if (data.wifi == null) {
                    return null;
                }

                return (
                    <Option
                        iconClassName={iconClassName}
                        labelClassName={labelClassName}
                        optionClassName={optionClassName}
                        icon={
                            <Icon>
                                <WiFiIcon />
                            </Icon>
                        }
                        label={
                            data.wifi
                                ? keyset.wiDashFiDashYes()
                                : keyset.wiDashFiDashNo()
                        }
                    />
                );
            }
            case 'power': {
                // eslint-disable-next-line no-eq-null
                if (data.power == null) {
                    return null;
                }

                return (
                    <Option
                        iconClassName={iconClassName}
                        labelClassName={labelClassName}
                        optionClassName={optionClassName}
                        icon={
                            <Icon>
                                <SocketIcon />
                            </Icon>
                        }
                        label={
                            data.power
                                ? keyset.inDashSeatDashPowerDashYes()
                                : keyset.inDashSeatDashPowerDashNo()
                        }
                    />
                );
            }
            case 'ife': {
                // eslint-disable-next-line no-eq-null
                if (data.ife == null) {
                    return null;
                }

                return (
                    <Option
                        iconClassName={iconClassName}
                        labelClassName={labelClassName}
                        optionClassName={optionClassName}
                        icon={
                            <Icon>
                                <EntertainmentIcon />
                            </Icon>
                        }
                        label={
                            data.ife
                                ? keyset.inDashFlightDashEntertainmentDashYes()
                                : keyset.inDashFlightDashEntertainmentDashNo()
                        }
                    />
                );
            }
        }
    };
