import React, {ReactNode, useCallback, useEffect, useMemo, useRef} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import isEmpty from 'lodash/isEmpty';
import {ParsedQuery} from 'query-string';
import isEqual from 'lodash/isEqual';
import {useHistory} from 'react-router-dom';

import {AVIA_SEARCH_FORM_NAME} from 'constants/reduxForm';
import {ANYWHERE_POINT_KEY} from 'projects/avia/constants/constants';

import {IWithClassName} from 'types/withClassName';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {
    IAviaSearchFormErrors,
    IAviaSearchFormPointField,
    IAviaSearchFormValues,
    TAviaSearchFormDateField,
} from 'projects/avia/components/SearchForm/types';
import {ECalendarType} from 'components/Calendar/types';
import {EIndexGoal} from 'utilities/metrika/types/goals';
import {EHistoryControlType} from 'server/loggers/avia/AviaActionLog/types/EHistoryControlType';
import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {IAviaSuggest} from 'types/avia/suggests/IAviaSuggest';
import {ISearchFormRenderTravelersBlockProps} from 'components/SearchForm/types/ISearchFormRenderTravelersBlockProps';
import {EAviaSuggestsApiItemType} from 'server/api/AviaSuggestsApi/types/TAviaGetSuggestsApiResponse';

import fillFormFromContext from 'reducers/avia/context/fillFormFromContext';
import {
    setAviaSearchFormAdultsFieldAction,
    setAviaSearchFormAviaClassFieldAction,
    setAviaSearchFormChildrenFieldAction,
    setAviaSearchFormEndDateFieldAction,
    setAviaSearchFormFromFieldAction,
    setAviaSearchFormInfantsFieldAction,
    setAviaSearchFormIsRoundTripFieldAction,
    setAviaSearchFormStartDateFieldAction,
    setAviaSearchFormToFieldAction,
} from 'reducers/avia/searchForm/actions';
import requestSearchSuggestsThunkAction from 'reducers/avia/searchSuggests/thunk/requestSearchSuggestsThunkAction';
import {fetchCalendarPrices as fetchAviaCalendarPricesAction} from 'reducers/avia/aviaCalendarPrices/actions';
import {
    incrementSearchFormChangesAction,
    setHistoryControlUsedAction,
} from 'reducers/avia/aviaLogMetrics/actions';
import {applyPreviousSearchAction} from 'reducers/avia/searchForm/thunk/applyPreviousSearchAction';
import {resetAviaSearch} from 'reducers/avia/search/results/resetSearch';
import {getAviaSearchHistory} from 'reducers/avia/personalization/thunkActions';
import {getHotelsSuggests} from 'reducers/hotels/personalization/thunkActions';

import searchFormSelector from 'selectors/avia/searchForm/searchFormSelector';
import {aviaPersonalizationSearchHistory} from 'selectors/avia/personalization/aviaPersonalizationSearchHistory';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import getSuggestTitleAndDescription from 'projects/avia/components/SearchForm/utilities/getSuggestTitleAndDescription';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModMobile} from 'utilities/stylesUtils';
import validateForm from 'projects/avia/components/SearchForm/utilities/validateForm';
import {reachGoal} from 'utilities/metrika';
import useDispatchedAction from 'utilities/hooks/useDispatchedAction';
import mapPreviousSearches from './utilities/mapPreviousSearches';
import {
    isOutdated,
    preparePreviousSearchData,
} from 'projects/avia/lib/previousSearches';
import {
    EAviaLinkType,
    getSearchFormUrl,
} from 'projects/avia/components/SearchForm/utilities/getSearchFormUrl';
import {isAnywhereKey} from 'utilities/strings/isAnywhereKey';
import {isTransportCrossSearch} from 'utilities/searchForm/isCrossSearch';
import {useExperiments} from 'utilities/hooks/useExperiments';

import * as i18n from 'i18n/avia-SearchForm';

import CommonSearchForm, {
    IBaseSearchFormProps,
    IOnDateClickParams,
} from 'components/SearchForm/SearchForm';
import AviaTravellers from 'projects/avia/components/SearchForm/components/AviaTravellers/AviaTravellers';
import RoundTrip from 'components/RoundTripToggle/RoundTripToggle';
import useRequestSuggestsWithThrottle from 'components/SearchForm/hooks/useRequestSuggestsWithThrottle';
import Intersperse from 'components/Intersperse/Intersperse';
import Text from 'components/Text/Text';
import DotSeparator from 'components/DotSeparator/DotSeparator';
import useSuggestInputFocusHandler from 'components/SearchForm/hooks/useSuggestInputFocusHandler';
import {
    ESuggestSource,
    ISuggestValue,
} from 'components/SearchSuggest/SearchSuggest';
import {IDatePickerFooterProps} from 'components/DatePicker/components/DatePickerFooter/DatePickerFooter';
import {useNewCalendarEnabled} from 'components/DatePicker/hooks/useNewCalendarEnabled';
import useCalendarConversionExpMetrics from 'components/DatePicker/hooks/useCalendarConversionExpMetrics';
import {TNameSuggest} from 'components/SearchForm/components/Suggest/Suggest';
import GlobeIcon from 'icons/16/Globe';

import cx from './SearchForm.scss';

const FIELD_NAMES = [
    ESearchFormFieldName.FROM,
    ESearchFormFieldName.TO,
    ESearchFormFieldName.START_DATE,
    ESearchFormFieldName.END_DATE,
];

interface ISearchFormProps extends IBaseSearchFormProps, IWithClassName {
    needToSetFromSuggestByGeoPosition?: boolean;
    hideDefaultGoal?: boolean;
    query?: ParsedQuery;
    useFilters?: boolean;
    useFiltersFromUrl?: boolean;
    onSubmit?(): void;
    customToPoint?: ISuggestValue<IAviaSuggest>;
    onChangeFromPoint?(value: ISuggestValue<IAviaSuggest>): void;
    onChangeToPoint?(value: ISuggestValue<IAviaSuggest>): void;
}

const SearchForm: React.FC<ISearchFormProps> = props => {
    const {
        className,
        needToSetFromSuggestByGeoPosition = false,
        hideDefaultGoal,
        query,
        useFilters,
        useFiltersFromUrl,
        fieldsRef,
        onSubmit,
        onInteract,
        isExpanded,
        onChangeFromPoint,
        customToPoint,
        onChangeToPoint,
        ...restProps
    } = props;

    const {
        fromField,
        toField,
        startDateField,
        endDateField,
        isRoundTripField,
        adultsField,
        childrenField,
        infantsField,
        aviaClassField,
        searchSuggests,
        calendar,
        filtersHash,
    } = useSelector(searchFormSelector);
    const {searchHistory} = useSelector(aviaPersonalizationSearchHistory);
    const canUseNewCalendar = useNewCalendarEnabled();
    const {reachCalendarInteraction, reachSearchInteraction} =
        useCalendarConversionExpMetrics(EAviaGoal);
    const {newDatePickerEnabled} = useExperiments();

    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const history = useHistory();
    const {
        location: {hash},
    } = history;

    const isInitialSuggestsRequestRef = useRef(true);
    const chosenPreviousSearch = useRef<IAviaSearchFormValues | null>(null);

    const setFromPoint = useImmutableCallback(
        (fieldValue: ISuggestValue<IAviaSuggest>) => {
            dispatch(setAviaSearchFormFromFieldAction(fieldValue));

            onChangeFromPoint?.(fieldValue);
        },
    );

    const setToPoint = useImmutableCallback(
        (fieldValue: ISuggestValue<IAviaSuggest>) => {
            dispatch(setAviaSearchFormToFieldAction(fieldValue));

            onChangeToPoint?.(fieldValue);
        },
    );

    useEffect(() => {
        if (customToPoint) {
            setToPoint(customToPoint);
        }
    }, [customToPoint, setToPoint]);

    const setStartDate = useDispatchedAction(
        setAviaSearchFormStartDateFieldAction,
        dispatch,
    );
    const setIsRoundTrip = useDispatchedAction(
        setAviaSearchFormIsRoundTripFieldAction,
        dispatch,
    );
    const setAdults = useDispatchedAction(
        setAviaSearchFormAdultsFieldAction,
        dispatch,
    );
    const setChildren = useDispatchedAction(
        setAviaSearchFormChildrenFieldAction,
        dispatch,
    );
    const setInfants = useDispatchedAction(
        setAviaSearchFormInfantsFieldAction,
        dispatch,
    );
    const setAviaClass = useDispatchedAction(
        setAviaSearchFormAviaClassFieldAction,
        dispatch,
    );

    const incrementSearchFormChanges = useCallback(
        () => dispatch(incrementSearchFormChangesAction()),
        [dispatch],
    );
    const setHistoryControlUsed = useDispatchedAction(
        setHistoryControlUsedAction,
        dispatch,
    );
    const applyStoredSearch = useDispatchedAction(
        applyPreviousSearchAction,
        dispatch,
    );

    const handleDateClick = useImmutableCallback(
        ({startDate, endDate}: IOnDateClickParams) => {
            incrementSearchFormChanges();

            if (startDate && !endDate) {
                reachGoal(EAviaGoal.AVIA_CALENDAR_WHEN_SELECTED);
            }

            if (startDate && endDate) {
                reachGoal(EAviaGoal.AVIA_CALENDAR_RETURN_SELECTED);
            }
        },
    );

    const setEndDate = useImmutableCallback(
        (date: TAviaSearchFormDateField) => {
            dispatch(setAviaSearchFormEndDateFieldAction(date));

            if (date && !isRoundTripField) {
                setIsRoundTrip(true);
            }
        },
    );

    const requestSuggests = useImmutableCallback(
        (from: IAviaSearchFormPointField, to: IAviaSearchFormPointField) => {
            // Не фетчим саджесты во время загрузки cross-search
            if (
                from.source === ESuggestSource.CROSS_SEARCH &&
                to.source === ESuggestSource.CROSS_SEARCH &&
                !from.selectedValue &&
                !to.selectedValue
            ) {
                return;
            }

            dispatch(
                requestSearchSuggestsThunkAction({
                    fromField: from,
                    toField: to,
                    needToSetByGeoPointIfPossible:
                        needToSetFromSuggestByGeoPosition &&
                        isInitialSuggestsRequestRef.current,
                }),
            );

            isInitialSuggestsRequestRef.current = false;
        },
    );

    const requestSuggestsWithThrottle =
        useRequestSuggestsWithThrottle(requestSuggests);

    const handleChangeRoundTrip = useImmutableCallback(
        ({target: {value}}: {target: {value: 'oneWay' | 'roundTrip'}}) => {
            const isRoundTrip = value === 'roundTrip';

            setIsRoundTrip(isRoundTrip);

            if (!isRoundTrip) {
                setEndDate(null);
            }

            const goal = isRoundTrip
                ? EAviaGoal.AVIA_CALENDAR_ROUNDTRIP
                : EAviaGoal.AVIA_CALENDAR_ONEWAY;

            if (!canUseNewCalendar) {
                reachGoal(goal);
            }
        },
    );

    const handleTravelersFocus = useImmutableCallback(() => {
        // Должно срабатывать после скрытия других попапов
        requestAnimationFrame(() => {
            onInteract?.(true);
        });
    });

    const handleTravelersBlur = useImmutableCallback(() => {
        onInteract?.(false);
    });

    const handleSelectSuggestItem = useCallback(
        (suggestValue: IAviaSuggest) => {
            const previousSearch = suggestValue?.previousSearch;

            if (previousSearch) {
                setHistoryControlUsed({type: EHistoryControlType.SUGGEST});
                applyStoredSearch(previousSearch);

                chosenPreviousSearch.current = previousSearch;

                reachGoal(EAviaGoal.PREVIOUS_SEARCHES_SUGGEST_CLICK);

                if (deviceType.isMobile && isOutdated(previousSearch)) {
                    setTimeout(() => {
                        fieldsRef?.current?.focusFieldByName(
                            ESearchFormFieldName.START_DATE,
                        );
                    });
                }

                return;
            } else if (isAnywhereKey(suggestValue.pointKey)) {
                reachGoal(EAviaGoal.AVIA_TO_ANYWHERE_SUGGEST_CLICK);
            }

            incrementSearchFormChanges();
        },
        [
            incrementSearchFormChanges,
            setHistoryControlUsed,
            applyStoredSearch,
            deviceType.isMobile,
            fieldsRef,
        ],
    );
    const getActiveFilters = useCallback(() => {
        if (useFilters) {
            return filtersHash;
        }

        if (useFiltersFromUrl) {
            return hash;
        }

        return '';
    }, [useFilters, useFiltersFromUrl, filtersHash, hash]);

    const handleCalendarShow = useCallback(
        (calendarType: ECalendarType) => {
            const oldStartCalendarOpened =
                !canUseNewCalendar && calendarType === ECalendarType.StartDate;
            const startCalendarOpened =
                oldStartCalendarOpened || canUseNewCalendar;

            if (!startCalendarOpened) {
                return;
            }

            reachGoal(EAviaGoal.CALENDAR_OPEN);

            if (toField.selectedValue && fromField.selectedValue) {
                reachCalendarInteraction();
            }
        },
        [canUseNewCalendar, fromField, toField, reachCalendarInteraction],
    );

    const handleSubmit = useImmutableCallback(
        ({
            handleTrySubmit,
            formErrors,
            setHasSubmitFailed,
        }: {
            handleTrySubmit(): void;
            formErrors: IAviaSearchFormErrors;
            setHasSubmitFailed(val: boolean): void;
        }) => {
            if (!hideDefaultGoal) {
                reachGoal(EIndexGoal.AVIA_SEARCH_BUTTON);
            }

            reachSearchInteraction();

            handleTrySubmit();

            if (!isEmpty(formErrors)) {
                return setHasSubmitFailed(true);
            }

            const fromSelectedValue = fromField.selectedValue;
            const toSelectedValue = toField.selectedValue;

            if (
                typeof fromSelectedValue === 'boolean' ||
                typeof toSelectedValue === 'boolean'
            ) {
                return;
            }

            const {url, linkType} = getSearchFormUrl({
                fromKey: fromSelectedValue.pointKey,
                toKey: toSelectedValue.pointKey,
                dateForward: startDateField || '',
                dateBackward: endDateField || '',
                adults: adultsField,
                children: childrenField,
                infants: infantsField,
                klass: aviaClassField,
                query,
                filterHash: getActiveFilters(),
            });

            if (!url) {
                return;
            }

            if (linkType === EAviaLinkType.SEARCH) {
                dispatch(resetAviaSearch());
                reachGoal(EAviaGoal.SEARCH_FROM_FORM);
            }

            history.push(url);

            onSubmit?.();

            if (
                isEqual(chosenPreviousSearch.current, {
                    [ESearchFormFieldName.FROM]: fromField,
                    [ESearchFormFieldName.TO]: toField,
                    [ESearchFormFieldName.START_DATE]: startDateField,
                    [ESearchFormFieldName.END_DATE]: endDateField,
                    isRoundTrip: isRoundTripField,
                    adults: adultsField,
                    children: childrenField,
                    infants: infantsField,
                    aviaClass: aviaClassField,
                })
            ) {
                reachGoal(EAviaGoal.PREVIOUS_SEARCHES_SUGGEST_SEARCH);

                chosenPreviousSearch.current = null;
            }

            if (isTransportCrossSearch(fromField, toField)) {
                reachGoal(EAviaGoal.CROSS_SEARCH_SUBMIT);
            }
        },
    );

    const requestCalendar = useImmutableCallback((type: ECalendarType) => {
        dispatch(fetchAviaCalendarPricesAction(type));
    });

    const renderTravellersBlock = useCallback(
        ({baseCx}: ISearchFormRenderTravelersBlockProps) => {
            return (
                <div className={cx('travellers')}>
                    <AviaTravellers
                        triggerClassName={baseCx(
                            'travellersTrigger',
                            'lastControl',
                        )}
                        triggerFocusClassName={baseCx(
                            'travellersTrigger_focus',
                        )}
                        tabIndex={isExpanded ? 0 : -1}
                        adultsCount={adultsField}
                        setAdults={setAdults}
                        childrenCount={childrenField}
                        setChildren={setChildren}
                        infantsCount={infantsField}
                        setInfants={setInfants}
                        aviaClass={aviaClassField}
                        setAviaClass={setAviaClass}
                        deviceType={deviceType}
                        onChange={incrementSearchFormChanges}
                        onFocus={handleTravelersFocus}
                        onBlur={handleTravelersBlur}
                    />
                </div>
            );
        },
        [
            adultsField,
            aviaClassField,
            infantsField,
            childrenField,
            deviceType,
            handleTravelersBlur,
            handleTravelersFocus,
            isExpanded,
            setAdults,
            setAviaClass,
            setInfants,
            setChildren,
            incrementSearchFormChanges,
        ],
    );

    const handleSuggestInputFocus = useSuggestInputFocusHandler(
        fromField,
        toField,
        requestSuggestsWithThrottle,
    );

    const roundTripToggleBlock = useMemo(() => {
        if (canUseNewCalendar) {
            return null;
        }

        return (
            <RoundTrip
                isRoundTrip={isRoundTripField}
                onChange={handleChangeRoundTrip}
            />
        );
    }, [handleChangeRoundTrip, isRoundTripField, canUseNewCalendar]);

    const footerBlockParams = useMemo((): IDatePickerFooterProps => {
        const messageNode =
            fromField.selectedValue && toField.selectedValue && startDateField
                ? i18n.pricesAlert()
                : undefined;

        const showButton = deviceType.isMobile && newDatePickerEnabled;

        const visible = Boolean(startDateField && (showButton || messageNode));

        const buttonText = endDateField
            ? i18n.selectButton()
            : i18n.oneWayTripButton();

        return {visible, messageNode, showButton, buttonText};
    }, [
        startDateField,
        endDateField,
        fromField.selectedValue,
        toField.selectedValue,
        deviceType.isMobile,
        newDatePickerEnabled,
    ]);

    const suggestPreviousFormValues = useMemo(
        () => mapPreviousSearches(searchHistory.items),
        [searchHistory.items],
    );

    const sortSuggestItems = useCallback((a: IAviaSuggest) => {
        // Показываем "Куда угодно" самым первым элементом саджестов
        if (a.type === EAviaSuggestsApiItemType.SPECIAL) {
            return -1;
        }

        return 0;
    }, []);

    useEffect(() => {
        dispatch(getAviaSearchHistory());
        dispatch(getHotelsSuggests());
    }, [dispatch]);

    useEffect(() => {
        requestSuggestsWithThrottle(fromField, toField);
    }, [fromField, requestSuggestsWithThrottle, toField]);

    useEffect(() => {
        dispatch(fillFormFromContext());
    }, [dispatch]);

    const prevSearchToAnywhereSuggestExists = useRef(false);

    const showSuggestHandler = useImmutableCallback(
        (type: TNameSuggest, items: IAviaSuggest[]) => {
            if (type !== ESearchFormFieldName.TO) {
                return;
            }

            const searchToAnywhereSuggestExists = Boolean(
                items.find(({pointKey}) => isAnywhereKey(pointKey)),
            );

            if (
                searchToAnywhereSuggestExists &&
                !prevSearchToAnywhereSuggestExists.current
            ) {
                reachGoal(EAviaGoal.AVIA_TO_ANYWHERE_SUGGEST_SHOW);
            }

            prevSearchToAnywhereSuggestExists.current =
                searchToAnywhereSuggestExists;
        },
    );

    return (
        <CommonSearchForm
            className={cx(deviceModMobile('root', deviceType), className)}
            searchFormName={AVIA_SEARCH_FORM_NAME}
            fieldsNames={FIELD_NAMES}
            fromField={fromField}
            toField={toField}
            startDateField={startDateField}
            endDateField={endDateField}
            hasEndDate
            fromSearchSuggests={searchSuggests[ESearchFormFieldName.FROM].value}
            toSearchSuggests={searchSuggests[ESearchFormFieldName.TO].value}
            previousFormValues={suggestPreviousFormValues}
            validateForm={validateForm}
            uniqueSuggestValueName="pointKey"
            getSuggestTitleAndDescription={getSuggestTitleAndDescription}
            fromPointPlaceholder={i18n.fromPlaceholder()}
            toPointPlaceholder={i18n.toPlaceholder()}
            startDatePlaceholder={i18n.startDatePlaceholder()}
            endDatePlaceholder={i18n.endDatePlaceholder()}
            datePickerHeaderBlock={roundTripToggleBlock}
            datePickerFooterBlockParams={footerBlockParams}
            setFromPoint={setFromPoint}
            setToPoint={setToPoint}
            setStartDate={setStartDate}
            setEndDate={setEndDate}
            onSubmit={handleSubmit}
            onSelectSuggestItem={handleSelectSuggestItem}
            onReverseClick={incrementSearchFormChanges}
            onDateClick={handleDateClick}
            isRoundTrip={canUseNewCalendar || isRoundTripField}
            renderTravellersBlock={renderTravellersBlock}
            calendarPrices={calendar}
            requestCalendar={requestCalendar}
            fieldsRef={fieldsRef}
            renderSuggestItemTitle={renderSuggestItemTitle}
            renderSuggestItemDescription={renderSuggestItemDescription}
            renderSuggestItemIcon={renderSuggestItemIcon}
            onInteract={onInteract}
            onSuggestInputFocus={handleSuggestInputFocus}
            canFocusNextField={canFocusNextField}
            resetRangeAfterFirstInteraction
            canAutoHideCalendar={
                canUseNewCalendar ? !deviceType.isMobile : undefined
            }
            onShowCalendar={handleCalendarShow}
            sortSuggestItems={sortSuggestItems}
            onShowSuggests={showSuggestHandler}
            {...prepareQaAttributes('search-form')}
            {...restProps}
        />
    );
};

function canFocusNextField(type: string, item: IAviaSuggest | false): boolean {
    const previousSearch = item && item?.previousSearch;

    return !previousSearch || isOutdated(previousSearch);
}

function renderSuggestItemTitle(item: IAviaSuggest): ReactNode {
    const {previousSearch} = item;

    if (previousSearch) {
        const {title} = preparePreviousSearchData(previousSearch);

        return (
            <Text size="m" weight="medium">
                {title}
            </Text>
        );
    }

    return getSuggestTitleAndDescription(item)?.title;
}

function renderSuggestItemDescription(item: IAviaSuggest): ReactNode {
    const {previousSearch} = item;

    if (previousSearch) {
        const {date, passengers} = preparePreviousSearchData(previousSearch);

        return (
            <div>
                <Intersperse
                    separator={<DotSeparator className={cx('separator')} />}
                >
                    {date ? (
                        <Text size="s" color="secondary">
                            {date}
                        </Text>
                    ) : null}
                    <Text size="s" color="secondary">
                        {passengers}
                    </Text>
                </Intersperse>
            </div>
        );
    }

    return getSuggestTitleAndDescription(item)?.description;
}

function renderSuggestItemIcon(item: IAviaSuggest): ReactNode {
    if (item.pointKey === ANYWHERE_POINT_KEY) return <GlobeIcon />;

    return null;
}

export default React.memo(SearchForm);
