import React, {PureComponent} from 'react';
import _noop from 'lodash/noop';

import {
    MAX_PASSENGERS_COUNT,
    MIN_ADULTS_COUNT,
    MIN_INFANTS_COUNT,
    MIN_CHILDREN_COUNT,
} from 'constants/avia';

import {IWithDeviceType} from 'types/withDeviceType';
import {EAviaClassType} from 'types/avia/EAviaClassType';
import {ECountFieldType} from 'types/common/ECountFieldType';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/avia';

import Travellers from 'components/Travellers/Travellers';
import RadioButtonSlide from 'components/RadioButtonSlide/RadioButtonSlide';

import cx from './AviaTravellers.scss';

const {TravellersCount} = Travellers;

const COUNT_FIELDS_TYPES = [
    ECountFieldType.ADULTS,
    ECountFieldType.CHILDREN,
    ECountFieldType.INFANTS,
];

const SELECTED_AVIA_CLASS_BY_TYPE = {
    [EAviaClassType.ECONOMY]:
        i18nBlock.componentsDotAviaTravellersDotEconomyClass(),
    [EAviaClassType.BUSINESS]:
        i18nBlock.componentsDotAviaTravellersDotBusinessClass(),
};

const AVIA_CLASS_TYPES: IChangeAviaClassParams[] = [
    {
        value: EAviaClassType.ECONOMY,
        label: i18nBlock.componentsDotAviaTravellersDotShortEconomyClass(),
    },
    {
        value: EAviaClassType.BUSINESS,
        label: i18nBlock.componentsDotAviaTravellersDotShortBusinessClass(),
    },
];

interface ICountField {
    value: number;
    onChange(value: number): void;
}

interface ITravellersCountFieldParams {
    field: ICountField;
    min: number;
    title: string;
    ageInfo: string;
}

interface IChangeAviaClassParams {
    label: string;
    value: EAviaClassType;
}

interface IAviaTravellersProps extends IWithDeviceType, IWithQaAttributes {
    adultsCount: number;
    setAdults(count: number): void;
    childrenCount: number;
    setChildren(count: number): void;
    infantsCount: number;
    setInfants(count: number): void;
    aviaClass: EAviaClassType;
    setAviaClass(aviaClass: EAviaClassType): void;
    tabIndex?: number;
    triggerRef: () => void;
    triggerClassName?: string;
    triggerFocusClassName?: string;
    onFocus(): void;
    onBlur(): void;
    onChange(): void;
}

class AviaTravellers extends PureComponent<IAviaTravellersProps> {
    static defaultProps = {
        triggerRef: _noop,
        triggerClassName: '',
        triggerFocusClassName: '',
        deviceType: {},
        onFocus: _noop,
        onBlur: _noop,
        onChange: _noop,
    };

    getTravellersCountFieldParamsByType(
        fieldType: ECountFieldType,
    ): ITravellersCountFieldParams {
        const {
            adultsCount,
            setAdults,
            childrenCount,
            setChildren,
            infantsCount,
            setInfants,
        } = this.props;

        switch (fieldType) {
            case ECountFieldType.ADULTS: {
                return {
                    field: {
                        value: adultsCount,
                        onChange: setAdults,
                    },
                    min: MIN_ADULTS_COUNT,
                    title: i18nBlock.componentsDotAviaTravellersDotAdultsTitle(),
                    ageInfo:
                        i18nBlock.componentsDotAviaTravellersDotAdultsAge(),
                };
            }

            case ECountFieldType.CHILDREN: {
                return {
                    field: {
                        value: childrenCount,
                        onChange: setChildren,
                    },
                    min: MIN_CHILDREN_COUNT,
                    title: i18nBlock.componentsDotAviaTravellersDotChildrenTitle(),
                    ageInfo:
                        i18nBlock.componentsDotAviaTravellersDotChildrenAge(),
                };
            }

            case ECountFieldType.INFANTS: {
                return {
                    field: {
                        value: infantsCount,
                        onChange: setInfants,
                    },
                    min: MIN_INFANTS_COUNT,
                    title: i18nBlock.componentsDotAviaTravellersDotBabyTitle(),
                    ageInfo: i18nBlock.componentsDotAviaTravellersDotBabyAge(),
                };
            }
        }
    }

    handleChangeTravellersCount =
        (type: ECountFieldType) =>
        (value: number): void => {
            const {onChange} = this.props;
            const {field} = this.getTravellersCountFieldParamsByType(type);

            if (type === ECountFieldType.ADULTS) {
                this.checkValidateInfantsCount(value);
            }

            field.onChange(value);
            onChange();
        };

    handleChangeAviaClass = (value: EAviaClassType): void => {
        const {setAviaClass, onChange} = this.props;

        setAviaClass(value);
        onChange();
    };

    checkValidateInfantsCount = (adultsValue: number): void => {
        const {
            field: {value, onChange},
        } = this.getTravellersCountFieldParamsByType(ECountFieldType.INFANTS);

        if (adultsValue < value) {
            onChange(adultsValue);
        }
    };

    getTotalTravellersCount(): number {
        return COUNT_FIELDS_TYPES.reduce((count, type) => {
            const {
                field: {value},
            } = this.getTravellersCountFieldParamsByType(type);

            return count + value;
        }, 0);
    }

    getSelectedAviaClassForTrigger(): string {
        const {aviaClass} = this.props;

        return SELECTED_AVIA_CLASS_BY_TYPE[aviaClass];
    }

    getMaxInfantsCount(maxCount: number): number {
        const {
            field: {value: adultsValue},
        } = this.getTravellersCountFieldParamsByType(ECountFieldType.ADULTS);

        return Math.min(adultsValue, maxCount);
    }

    renderTravellersButtonContent = (): React.ReactNode => {
        const totalTravellers = this.getTotalTravellersCount();
        const selectedAviaClass = this.getSelectedAviaClassForTrigger();
        const travellersValue =
            i18nBlock.componentsDotAviaTravellersDotTravellers({
                count: totalTravellers,
            });

        return (
            <div className={cx('travellersButton')}>
                <div className={cx('travellersButtonValue')}>
                    {travellersValue}
                </div>
                <div className={cx('selectedAviaClass')}>
                    {selectedAviaClass}
                </div>
            </div>
        );
    };

    renderTravellersCount = (type: ECountFieldType): React.ReactNode => {
        const {
            field: {value},
            min,
        } = this.getTravellersCountFieldParamsByType(type);
        const totalTravellers = this.getTotalTravellersCount();
        const remainTravellers = MAX_PASSENGERS_COUNT - totalTravellers;
        const maxCount = value + remainTravellers;
        const isInfantField = type === ECountFieldType.INFANTS;
        const max = isInfantField
            ? this.getMaxInfantsCount(maxCount)
            : maxCount;
        const {title, ageInfo} = this.getTravellersCountFieldParamsByType(type);

        return (
            <TravellersCount
                className={cx('travellersCount')}
                key={type}
                value={value}
                min={min}
                max={max}
                label={title}
                hint={ageInfo}
                onChange={this.handleChangeTravellersCount(type)}
                {...prepareQaAttributes({
                    parent: 'travellersCount',
                    key: type,
                })}
            />
        );
    };

    renderAviaClassToggle(): React.ReactNode {
        const {aviaClass} = this.props;

        return (
            <RadioButtonSlide
                value={aviaClass}
                onChange={this.handleChangeAviaClass}
            >
                {AVIA_CLASS_TYPES.map(({value, label}) => (
                    <RadioButtonSlide.Option
                        key={value}
                        value={value}
                        {...prepareQaAttributes({
                            parent: 'avia-travellers-search-button-group',
                            current: value,
                        })}
                    >
                        {label}
                    </RadioButtonSlide.Option>
                ))}
            </RadioButtonSlide>
        );
    }

    render(): React.ReactNode {
        const {
            tabIndex,
            triggerRef,
            deviceType,
            triggerClassName,
            triggerFocusClassName,
            onFocus,
            onBlur,
        } = this.props;
        const {isMobile} = deviceType;

        return (
            <Travellers
                tabIndex={tabIndex}
                deviceType={deviceType}
                triggerRef={triggerRef}
                isModalView={isMobile}
                type="travellers"
                triggerClassName={triggerClassName}
                triggerFocusClassName={triggerFocusClassName}
                buttonComponent={this.renderTravellersButtonContent}
                onFocus={onFocus}
                onBlur={onBlur}
                canToggle
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'travellers',
                })}
            >
                <React.Fragment>
                    {COUNT_FIELDS_TYPES.map(this.renderTravellersCount)}
                    {this.renderAviaClassToggle()}
                </React.Fragment>
            </Travellers>
        );
    }
}

export default AviaTravellers;
