import {ParsedQuery} from 'query-string';

import {EAviaClassType} from 'types/avia/EAviaClassType';
import {IPassengers} from 'projects/avia/pages/AviaBooking/types/IPassengers';

import {hashToFilterValues} from 'projects/avia/lib/search/filters/converters';
import {getUtmQueryParamsByQuery} from 'utilities/getUtmQueryParams/getUtmQueryParams';
import {isSettlementKey} from 'utilities/strings/isSettlementKey';
import {isStationKey} from 'utilities/strings/isStationKey';
import isDateRobot from 'utilities/dateUtils/isDateRobot';
import {isCountryKey} from 'utilities/strings/isCountryKey';
import {getSearchUrl} from 'projects/avia/lib/urls/getSearchUrl';
import {getSearchToCountryUrl} from 'projects/avia/lib/urls/getSearchToCountryUrl';
import {isAnywhereKey} from 'utilities/strings/isAnywhereKey';
import {getSearchToAnywhereUrl} from 'projects/avia/lib/urls/getSearchToAnywhereUrl';

/**
 * Тип ссылки (на какую страницу ведет)
 */
export enum EAviaLinkType {
    SEARCH = 'SEARCH',
    SEARCH_TO_COUNTRY = 'SEARCH_TO_COUNTRY',
    SEARCH_TO_ANYWHERE = 'SEARCH_TO_ANYWHERE',
}

interface IGetSearchFormUrlParams {
    fromKey: string; // pointKey
    toKey: string; // pointKey
    dateForward: string;
    dateBackward: string;
    adults: number;
    children: number;
    infants: number;
    klass: EAviaClassType;

    filterHash?: string;
    // Из query будут извлечены utm-метки и прокинуты в урл
    query?: ParsedQuery;
}

interface IGetSearchFormUrlResponse {
    url: string | null;
    linkType: EAviaLinkType | null;
}

const failAnswer: IGetSearchFormUrlResponse = {url: null, linkType: null};

/**
 * Возвращает урл, на который нужно перейти, после сабмита формы поиска авиа
 */
export function getSearchFormUrl({
    fromKey,
    toKey,
    dateForward,
    dateBackward: dateBackwardString,
    adults,
    children,
    infants,
    klass,
    filterHash,
    query,
}: IGetSearchFormUrlParams): IGetSearchFormUrlResponse {
    if (!isSettlementKey(fromKey) && !isStationKey(fromKey)) {
        return failAnswer;
    }

    if (!isDateRobot(dateForward)) {
        return failAnswer;
    }

    const dateBackward = isDateRobot(dateBackwardString)
        ? dateBackwardString
        : undefined;

    const utm = query ? getUtmQueryParamsByQuery(query) : undefined;
    const passengers: IPassengers = {
        adults,
        children,
        infants,
    };

    if (isSettlementKey(toKey) || isStationKey(toKey)) {
        const filters = filterHash ? hashToFilterValues(filterHash) : undefined;

        return {
            linkType: EAviaLinkType.SEARCH,
            url: getSearchUrl({
                fromKey,
                toKey,
                dateForward,
                dateBackward,
                klass,
                passengers,
                filters,
                utm,
            }),
        };
    } else if (isCountryKey(toKey)) {
        return {
            linkType: EAviaLinkType.SEARCH_TO_COUNTRY,
            url: getSearchToCountryUrl({
                fromKey,
                toKey,
                dateForward,
                dateBackward,
                passengers,
                klass,
                utm,
            }),
        };
    } else if (isAnywhereKey(toKey)) {
        return {
            linkType: EAviaLinkType.SEARCH_TO_ANYWHERE,
            url: getSearchToAnywhereUrl({
                fromKey,
                dateForward,
                dateBackward,
                passengers,
                klass,
                utm,
            }),
        };
    }

    return failAnswer;
}
