import moment from 'moment-timezone';

import {
    IAviaSearchFormErrors,
    IAviaSearchFormPointField,
    IAviaSearchFormValues,
    TAviaSearchFormDateField,
} from 'projects/avia/components/SearchForm/types';
import {IAviaSuggest} from 'types/avia/suggests/IAviaSuggest';
import {
    ESearchFormFieldName,
    ISearchFormSuggests,
} from 'components/SearchForm/types';

import {isStationKey} from 'utilities/strings/isStationKey';
import {isSettlementKey} from 'utilities/strings/isSettlementKey';

import * as i18n from 'i18n/avia-SearchForm';

import {ESuggestSource} from 'components/SearchSuggest/SearchSuggest';

const isStationOrSettlement = ({
    selectedValue,
}: IAviaSearchFormPointField): boolean => {
    if (selectedValue === false) {
        return false;
    }

    const {pointKey} = selectedValue;

    if (isStationKey(pointKey) || isSettlementKey(pointKey)) {
        return true;
    }

    return false;
};

const isEmptyDate = (date: TAviaSearchFormDateField): boolean => !date;

const isEmptyDirectionValue = ({
    inputValue,
}: IAviaSearchFormPointField): boolean => inputValue === '';

const isDirectionNotFound = ({
    inputValue,
    selectedValue,
}: IAviaSearchFormPointField): boolean =>
    inputValue !== '' && selectedValue === false;

const isEmptySelectedValue = ({
    selectedValue,
}: IAviaSearchFormPointField): boolean => selectedValue === false;

const isInvalidRangeDates = (
    startDate: TAviaSearchFormDateField,
    endDate: TAviaSearchFormDateField,
): boolean => {
    if (!startDate || !endDate) {
        return false;
    }

    return moment(endDate).isBefore(startDate);
};

const isEqualDirectionsCities = (
    fromField: IAviaSearchFormPointField,
    toField: IAviaSearchFormPointField,
): boolean => {
    const {selectedValue: fromFieldSelectedValue} = fromField;
    const {selectedValue: toFieldSelectedValue} = toField;

    if (!fromFieldSelectedValue || !toFieldSelectedValue) {
        return false;
    }

    const {pointKey: fromFieldPointKey, parentId: fromFieldParentId} =
        fromFieldSelectedValue;
    const {pointKey: toFieldPointKey, parentId: toFieldParentId} =
        toFieldSelectedValue;

    return Boolean(
        fromFieldPointKey === toFieldPointKey ||
            fromFieldPointKey === toFieldParentId ||
            fromFieldParentId === toFieldPointKey ||
            (fromFieldParentId &&
                toFieldParentId &&
                fromFieldParentId === toFieldParentId),
    );
};

const isEqualDirectionsAirports = (
    fromField: IAviaSearchFormPointField,
    toField: IAviaSearchFormPointField,
): boolean => {
    const {selectedValue: fromFieldSelectedValue} = fromField;
    const {selectedValue: toFieldSelectedValue} = toField;

    if (
        !fromFieldSelectedValue ||
        !toFieldSelectedValue ||
        !fromFieldSelectedValue.pointCode ||
        !toFieldSelectedValue.pointCode
    ) {
        return false;
    }

    return fromFieldSelectedValue.pointCode === toFieldSelectedValue.pointCode;
};

/**
 * Валидация существует, потому что значение из cross-search подставляется еще до загрузки саджестов,
 * и перед поиском надо провалидировать, есть ли на самом деле такой саджест или нет
 */
const isUnknownCrossSearch = (
    field: IAviaSearchFormPointField,
    suggests: IAviaSuggest[] | null,
): boolean => {
    const {selectedValue, source} = field;

    if (!selectedValue || source !== ESuggestSource.CROSS_SEARCH || !suggests) {
        return false;
    }

    return suggests.every(suggest => {
        return suggest.pointKey !== selectedValue.pointKey;
    });
};

export default (
    fieldsValues: IAviaSearchFormValues,
    suggests: ISearchFormSuggests<IAviaSuggest>,
): IAviaSearchFormErrors => {
    if (!fieldsValues) {
        return {};
    }

    const {from, to, startDate, endDate} = fieldsValues;
    const errors: IAviaSearchFormErrors = {};

    if (
        isDirectionNotFound(from) ||
        isUnknownCrossSearch(from, suggests[ESearchFormFieldName.FROM])
    ) {
        errors.from = [i18n.validateUnknownFromDirection()];
    } else if (isEmptyDirectionValue(from)) {
        errors.from = [i18n.validateFromFieldRequired()];
    } else if (isEmptySelectedValue(from)) {
        errors.from = [i18n.validateNeedSelectFromField()];
    } else if (!isStationOrSettlement(from)) {
        errors.from = [i18n.validateFromFieldCountryType()];
    }

    if (
        isDirectionNotFound(to) ||
        isUnknownCrossSearch(to, suggests[ESearchFormFieldName.TO])
    ) {
        errors.to = [i18n.validateUnknownToDirection()];
    } else if (isEmptyDirectionValue(to)) {
        errors.to = [i18n.validateToFieldRequired()];
    } else if (isEmptySelectedValue(to)) {
        errors.to = [i18n.validateNeedSelectToField()];
    } else if (isEqualDirectionsCities(from, to)) {
        errors.to = [i18n.validateEqualDirections()];
    } else if (isEqualDirectionsAirports(from, to)) {
        errors.to = [i18n.validateEqualAirports()];
    }

    if (isEmptyDate(startDate)) {
        errors.startDate = [i18n.validateStartDateRequired()];
    }

    if (isInvalidRangeDates(startDate, endDate)) {
        errors.endDate = [i18n.validateInvalidRangeDates()];
    }

    return errors;
};
