import React, {useCallback} from 'react';
import {useSelector} from 'react-redux';

import {HEIGHT_IMAGES} from 'projects/avia/constants/constants';

import {IImage} from 'types/common/IImage';
import {TSettlementKey} from 'types/PointKey';
import {IWithClassName} from 'types/withClassName';
import {IExpiringPrice} from 'types/common/price/IExpiredPrice';
import EAsyncStatus from 'types/common/EAsyncStatus';
import {IInlineSearchInfo} from 'types/avia/landing/IInlineSearchInfo';

import {StoreInterface} from 'reducers/storeTypes';

import {getOptimalImage} from 'utilities/images/getOptimalImage';

import NoHotelPhotoM from 'icons/illustrations/NoHotelPhoto/NoHotelPhotoM';
import NoHotelPhotoS from 'icons/illustrations/NoHotelPhoto/NoHotelPhotoS';
import Price from 'components/Price/Price';
import Spinner from 'components/Spinner/Spinner';

import cx from './Settlement.scss';

export interface ISettlement extends IWithClassName {
    /**
     * Ключ города прибытия
     * @example 'c213'
     */
    cityKey: TSettlementKey;
    /**
     * Название города прибытия в именительном падеже
     * @example 'Екатеринбург'
     */
    title: string;
    /** Дополнительная информация */
    description?: string;
    /**
     * Массив массивов изображений города прибытия.
     * То есть images[0] содержит массив с разными размерами первого изображения
     */
    images: IImage[][];
    /** Минимальная ширина изображения в css-пикселях (зависит от верстки в месте использования компонента) */
    minWidth: number;
    /**
     * Размер компонента.
     * Обычно 'm' для десктопа и 's' для тача.
     */
    size: 'm' | 's';
    /** Цена из priceIndex */
    price: IExpiringPrice | null;
    /** Селектор, который вытаскивает данные об инлайн-поиске для города */
    inlineSearchDataForCitySelector: (
        state: StoreInterface,
        cityKey: TSettlementKey,
    ) => IInlineSearchInfo | undefined;
    onClick?(cityKey: TSettlementKey): void;
}

/** Компонент отображения города для страницы поиска до страны и "куда угодно" */
const SettlementLink: React.FC<ISettlement> = ({
    cityKey,
    title,
    description,
    images,
    minWidth,
    size,
    price: priceFromPriceIndex,
    inlineSearchDataForCitySelector,
    onClick,
    className,
}) => {
    const inlineSearchInfo = useSelector((state: StoreInterface) =>
        inlineSearchDataForCitySelector(state, cityKey),
    );
    const price =
        inlineSearchInfo && inlineSearchInfo.status !== EAsyncStatus.ERROR
            ? inlineSearchInfo.price || priceFromPriceIndex
            : priceFromPriceIndex;

    const height = HEIGHT_IMAGES[size];
    const imageForNormalDisplay = images[0]
        ? getOptimalImage(images[0], minWidth, height)
        : undefined;
    const imageForRetinaDisplay = images[0]
        ? getOptimalImage(images[0], minWidth * 2, height * 2)
        : undefined;

    const BackgroundImage = size === 'm' ? NoHotelPhotoM : NoHotelPhotoS;

    const onClickHandler = useCallback(() => {
        onClick?.(cityKey);
    }, [cityKey, onClick]);

    return (
        <div
            className={cx('root', className, {
                size_s: size === 's',
                size_m: size === 'm',
            })}
            onClick={onClickHandler}
        >
            <div className={cx('imageContainer')}>
                <div className={cx('backgroundImageContainer')}>
                    <BackgroundImage className={cx('backgroundImage')} />
                </div>

                {imageForNormalDisplay !== undefined && (
                    <img
                        className={cx('image')}
                        alt={title}
                        src={imageForNormalDisplay.url}
                        srcSet={
                            imageForRetinaDisplay !== undefined &&
                            imageForRetinaDisplay.url !==
                                imageForNormalDisplay.url
                                ? `${imageForNormalDisplay.url} 1x, ${imageForRetinaDisplay.url} 2x`
                                : undefined
                        }
                    />
                )}
            </div>
            <div className={cx('title')}>{title}</div>
            {description && (
                <div className={cx('description')}>{description}</div>
            )}
            <div className={cx('priceContainer')}>
                {price && (
                    <Price
                        className={cx('price')}
                        value={price.value}
                        currency={price.currency}
                        isFrom={true}
                    />
                )}

                {inlineSearchInfo &&
                    inlineSearchInfo.status === EAsyncStatus.LOADING && (
                        <Spinner size="xxs" />
                    )}
            </div>
        </div>
    );
};

export default React.memo(SettlementLink);
