import {Component, ComponentType, ReactNode} from 'react';

type TStateReader<Value> = ComponentType<{
    children: (value: Value) => ReactNode;
}>;

type TStateSetter<Value> = (value: Value) => void;

/**
 * Создает пару Компонента и функция сеттер.
 *
 * Компонента передает в дочерние элементы дефолтное значение или последнее значение
 * установленное функцией сеттером.
 *
 * @param {Value} defaultValue - дефолтное значение
 *
 * @returns {[TStateReader<Value>, TStateSetter<Value>]} - возвращает
 * пару из компоненты и функции позволяющей засетить значение
 *
 * @template Value
 */
export function createStateContainer<Value>(
    defaultValue: Value,
): [TStateReader<Value>, TStateSetter<Value>] {
    const listeners = new Set<(value: Value) => any>();

    const subscribe = (listener: (value: Value) => any) => {
        listeners.add(listener);

        return () => listeners.delete(listener);
    };

    const setState = (value: Value): void => {
        for (const listener of listeners) {
            listener(value);
        }
    };

    const lastValue = defaultValue;

    class StateReader extends Component<
        {children: (value: Value) => ReactNode},
        {value: Value}
    > {
        private unsubscribe: (() => any) | null = null;

        state = {
            value: lastValue,
        };

        componentDidMount(): void {
            this.unsubscribe = subscribe(value => this.setState({value}));
        }

        componentWillUnmount(): void {
            if (this.unsubscribe) {
                this.unsubscribe();
            }
        }

        render(): ReactNode {
            return this.props.children(this.state.value);
        }
    }

    return [StateReader, setState];
}
