import React, {useMemo} from 'react';
import upperFirst from 'lodash/upperFirst';

import {
    IVariantPriceInfoDTO,
    IYandexPlusPromoCampaign,
} from 'server/api/AviaBookingApi/types/IBookingVariantsResponse';
import {ISegmentFare} from 'server/api/AviaBookingApi/types/ISegmentFare';
import {isNotNullable} from 'types/utilities';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlockAviaPlus from 'i18n/avia-plusPromo2021';
import * as i18nBlock from 'i18n/avia-AviaBooking-TariffsModal';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import YandexPlusSignColorGradientS from 'icons/illustrations/YandexPlusSignColorGradient/YandexPlusSignColorGradientS';
import Price from 'components/Price/Price';
import Button from 'components/Button/Button';

import cx from './TariffHeading.scss';

interface ITariffHeadingProps extends IWithQaAttributes {
    fareTerms: ISegmentFare;
    priceInfo: IVariantPriceInfoDTO | undefined;
    variantId: string;
    isActive: boolean;
    isPromo2020: boolean;
    plusPromo2021: IYandexPlusPromoCampaign | undefined;
    hasPlusRow?: boolean;
    onChange?(token: string): void;
}

const TariffHeading: React.FC<ITariffHeadingProps> = props => {
    const {
        fareTerms,
        priceInfo,
        variantId,
        isActive,
        isPromo2020,
        plusPromo2021,
        hasPlusRow,
        onChange,
    } = props;

    const plusPoints =
        plusPromo2021?.enabled && isNotNullable(plusPromo2021?.totalPlusPoints)
            ? plusPromo2021?.totalPlusPoints
            : undefined;

    const handleClick = useMemo(() => {
        if (!onChange) {
            return undefined;
        }

        return (): void => {
            onChange(variantId);
        };
    }, [onChange, variantId]);

    return (
        <Flex flexDirection="column" alignItems="center" y="5">
            <Box
                below="2"
                {...prepareQaAttributes({parent: props, current: 'name'})}
            >
                {upperFirst(fareTerms.tariffGroupName.toLowerCase())}
            </Box>

            {priceInfo && (
                <Box
                    textSize="l"
                    textWeight="medium"
                    below={plusPoints ? 2 : 3}
                    above={isPromo2020 ? 1 : 0}
                >
                    <Price
                        {...priceInfo.total}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'price',
                        })}
                    />
                </Box>
            )}

            {plusPoints && (
                <Box below="3">
                    <TextWithIcon
                        iconLeft={YandexPlusSignColorGradientS}
                        iconLeftProps={{width: 16, height: 16}}
                        spaceBetween="1"
                        text={i18nBlockAviaPlus.bookingTariffTablePlusRefund({
                            count: plusPoints,
                        })}
                    />
                </Box>
            )}

            {handleClick && (
                <Button
                    size="m-inset"
                    className={cx({
                        button_hasPlusRow: !plusPoints && hasPlusRow,
                    })}
                    theme={isPromo2020 ? 'primary' : 'secondary'}
                    disabled={isActive}
                    onClick={handleClick}
                    {...prepareQaAttributes({parent: props, current: 'button'})}
                >
                    {isActive ? i18nBlock.current() : i18nBlock.select()}
                </Button>
            )}
        </Flex>
    );
};

export default TariffHeading;
