import React, {ReactNode, FunctionComponent} from 'react';
import {useSelector} from 'react-redux';

import {isNotNull} from 'types/utilities';
import {ITermValue} from 'server/api/AviaBookingApi/types/ITermValue';
import {TermAvailability} from 'server/api/AviaBookingApi/types/TermAvailability';
import {IFlightTariff, IFlightTariffs} from '../../types/tariffTypes';

import {aviaAeroflotPlusSelector} from 'projects/avia/selectors/aviaAeroflotPlusEnabledSelector';

import {formatTermType} from 'projects/avia/lib/flightFareTerms/formatters';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Heading from 'components/Heading/Heading';
import {FlightTariffTable} from 'projects/avia/components/FlightTariffTable/FlightTariffTable';
import BaggageFormatter from 'projects/avia/components/BaggageFormatter/BaggageFormatter';
import {FlightFareAvailabilityTerm} from 'projects/avia/components/FlightFareAvailabilityTerm/FlightFareAvailabilityTerm';
import TariffHeading from '../../components/TariffHeading/TariffHeading';

const TARIFF_QA = 'tariff';

interface IFareTermRowProps extends IWithQaAttributes {
    tariffs: Map<string, IFlightTariff>;
    termKey: string;
    children(term: ITermValue, termQa: IWithQaAttributes): React.ReactNode;
    label?: string;
    isActive?: boolean;
}

const FareTermRow: FunctionComponent<IFareTermRowProps> = props => {
    const {tariffs, label, termKey, children} = props;

    return (
        <FlightTariffTable.Row borders={{bottom: true, top: true}}>
            <FlightTariffTable.Cell>{label}</FlightTariffTable.Cell>
            {[...tariffs].map(([variantId, {fareTerms}]) => {
                const term = fareTerms.terms[termKey];

                const variantQa = prepareQaAttributes({
                    parent: props,
                    current: TARIFF_QA,
                    key: variantId,
                });
                const termQa = prepareQaAttributes({
                    parent: variantQa,
                    current: termKey,
                });

                return (
                    <FlightTariffTable.Cell key={variantId}>
                        <Flex justifyContent="center" y="3">
                            {term ? (
                                children(term, termQa)
                            ) : (
                                <FlightFareAvailabilityTerm
                                    availability={
                                        TermAvailability.NOT_AVAILABLE
                                    }
                                    {...prepareQaAttributes(termQa)}
                                />
                            )}
                        </Flex>
                    </FlightTariffTable.Cell>
                );
            })}
        </FlightTariffTable.Row>
    );
};

interface IVariantsTariffTableDesktopProps extends IWithQaAttributes {
    flightTariffs: Map<string, IFlightTariffs>;
    activeColumn: number;
    onChange?(token: string): void;
}

export const VariantsTariffTableDesktop: React.FC<IVariantsTariffTableDesktopProps> =
    props => {
        const {activeColumn, flightTariffs, onChange} = props;

        const aviaAeroflotPlus = useSelector(aviaAeroflotPlusSelector);

        return (
            <Box between={8}>
                {[...flightTariffs].map(([flightId, {title, tariffs}]) => {
                    const hasPlusRow =
                        aviaAeroflotPlus &&
                        [...tariffs].some(([, tariff]) =>
                            isNotNull(
                                tariff.promoCampaigns?.plusPromo2021
                                    ?.totalPlusPoints,
                            ),
                        );

                    return (
                        <Box key={flightId} between={5}>
                            <Heading
                                level="2"
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'title',
                                })}
                            >
                                {title}
                            </Heading>

                            <FlightTariffTable
                                activeColumn={
                                    activeColumn === -1
                                        ? undefined
                                        : activeColumn + 1
                                }
                            >
                                <FlightTariffTable.Row borders={{bottom: true}}>
                                    <FlightTariffTable.Cell />
                                    {[...tariffs].map(
                                        (
                                            [
                                                variantId,
                                                {
                                                    fareTerms,
                                                    priceInfo,
                                                    promoCampaigns,
                                                },
                                            ],
                                            idx,
                                        ) => {
                                            const variantQa =
                                                prepareQaAttributes({
                                                    parent: props,
                                                    key: variantId,
                                                    current: TARIFF_QA,
                                                });

                                            return (
                                                <FlightTariffTable.Cell
                                                    key={variantId}
                                                >
                                                    <TariffHeading
                                                        fareTerms={fareTerms}
                                                        priceInfo={priceInfo}
                                                        variantId={variantId}
                                                        isActive={
                                                            idx === activeColumn
                                                        }
                                                        // TODO: white-monday завязаться на реальный признак с бэка
                                                        isPromo2020={Boolean(
                                                            promoCampaigns
                                                                ?.promo2020
                                                                ?.eligible,
                                                        )}
                                                        plusPromo2021={
                                                            aviaAeroflotPlus
                                                                ? promoCampaigns?.plusPromo2021
                                                                : undefined
                                                        }
                                                        hasPlusRow={hasPlusRow}
                                                        onChange={onChange}
                                                        {...prepareQaAttributes(
                                                            variantQa,
                                                        )}
                                                    />
                                                </FlightTariffTable.Cell>
                                            );
                                        },
                                    )}
                                </FlightTariffTable.Row>

                                <FareTermRow
                                    tariffs={tariffs}
                                    termKey="carryOn"
                                    label={formatTermType('carryOn')}
                                    {...prepareQaAttributes(props)}
                                >
                                    {(carryOn, termQa): ReactNode => (
                                        <BaggageFormatter
                                            {...carryOn}
                                            {...termQa}
                                        />
                                    )}
                                </FareTermRow>

                                <FareTermRow
                                    tariffs={tariffs}
                                    termKey="baggage"
                                    label={formatTermType('baggage')}
                                    {...prepareQaAttributes(props)}
                                >
                                    {(baggage, termQa): ReactNode => (
                                        <BaggageFormatter
                                            showNoBaggage
                                            places={baggage.places}
                                            weight={baggage.weight}
                                            {...termQa}
                                        />
                                    )}
                                </FareTermRow>

                                <FareTermRow
                                    tariffs={tariffs}
                                    termKey="refundable"
                                    label={formatTermType('refundable')}
                                    {...prepareQaAttributes(props)}
                                >
                                    {(term, termQa): ReactNode => (
                                        <FlightFareAvailabilityTerm
                                            {...term}
                                            {...termQa}
                                        />
                                    )}
                                </FareTermRow>

                                <FareTermRow
                                    tariffs={tariffs}
                                    termKey="changingCarriage"
                                    label={formatTermType('changingCarriage')}
                                    {...prepareQaAttributes(props)}
                                >
                                    {(term, termQa): ReactNode => (
                                        <FlightFareAvailabilityTerm
                                            {...term}
                                            {...termQa}
                                        />
                                    )}
                                </FareTermRow>

                                <FareTermRow
                                    tariffs={tariffs}
                                    termKey="miles"
                                    label={formatTermType('miles')}
                                    {...prepareQaAttributes(props)}
                                >
                                    {(miles, termQa): ReactNode => (
                                        <span {...termQa}>{miles.miles}</span>
                                    )}
                                </FareTermRow>

                                <FlightTariffTable.Row borders={{top: true}}>
                                    <FlightTariffTable.Cell />
                                    {[...tariffs].map((_, idx) => (
                                        <FlightTariffTable.Cell key={idx}>
                                            <Box below="5" />
                                        </FlightTariffTable.Cell>
                                    ))}
                                </FlightTariffTable.Row>
                            </FlightTariffTable>
                        </Box>
                    );
                })}
            </Box>
        );
    };
