import React, {FunctionComponent, ReactNode} from 'react';
import upperFirst from 'lodash/upperFirst';
import {useSelector} from 'react-redux';
import {useInView} from 'react-intersection-observer';

import {ITermValue} from 'server/api/AviaBookingApi/types/ITermValue';
import {TermAvailability} from 'server/api/AviaBookingApi/types/TermAvailability';
import {isNotNull} from 'types/utilities';
import {TBookingPromoCampaigns} from 'server/api/AviaBookingApi/types/IBookingVariantsResponse';
import {IFlightTariffs} from '../../types/tariffTypes';

import {aviaAeroflotPlusSelector} from 'projects/avia/selectors/aviaAeroflotPlusEnabledSelector';

import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {formatTermType} from 'projects/avia/lib/flightFareTerms/formatters';

import * as i18nBlock from 'i18n/avia-AviaBooking-TariffsModal';
import * as i18nBlockAviaPlus from 'i18n/avia-plusPromo2021';

import Box from 'components/Box/Box';
import Card from 'components/Card/Card';
import Heading from 'components/Heading/Heading';
import Price from 'components/Price/Price';
import Button from 'components/Button/Button';
import {FlightTariffTable} from 'projects/avia/components/FlightTariffTable/FlightTariffTable';
import {FlightFareAvailabilityTerm} from 'projects/avia/components/FlightFareAvailabilityTerm/FlightFareAvailabilityTerm';
import BaggageFormatter from 'projects/avia/components/BaggageFormatter/BaggageFormatter';
import Flex from 'components/Flex/Flex';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import YandexPlusSignColorGradientS from 'icons/illustrations/YandexPlusSignColorGradient/YandexPlusSignColorGradientS';

import cx from './VariantsTariffTableMobile.scss';

interface IFareTermRowProps {
    term: ITermValue;
    children(term: ITermValue): React.ReactNode;
    label?: string;
    bottomBorder?: boolean;
}

const FareTermRow: FunctionComponent<IFareTermRowProps> = ({
    term,
    label,
    children,
    bottomBorder = true,
}) => {
    return (
        <FlightTariffTable.Row borders={{bottom: bottomBorder, top: true}}>
            <FlightTariffTable.Cell>
                <Flex justifyContent="space-between" y="2">
                    <span>{label}</span>
                    {term && children(term)}
                    {!term && (
                        <FlightFareAvailabilityTerm
                            availability={TermAvailability.NOT_AVAILABLE}
                        />
                    )}
                </Flex>
            </FlightTariffTable.Cell>
        </FlightTariffTable.Row>
    );
};

interface IVariantsTariffTableMobileProps extends IWithQaAttributes {
    flightTariffs: Map<string, IFlightTariffs>;
    activeColumn: number;
    onChange?(token: string): void;
}

export const VariantsTariffTableMobile: React.FC<IVariantsTariffTableMobileProps> =
    props => {
        const {activeColumn, onChange, flightTariffs} = props;
        const containerRef = React.useRef<HTMLElement>(null);
        const activeCardRef = React.useRef<HTMLDivElement>(null);
        const [ref, inView] = useInView();

        React.useEffect(() => {
            if (inView && activeCardRef.current && containerRef.current) {
                const containerWidth = containerRef.current.offsetWidth;
                const cardWidth = activeCardRef.current.offsetWidth;
                const gap = (containerWidth - cardWidth) / 2;

                containerRef.current.scrollLeft = Math.max(
                    activeCardRef.current.offsetLeft - gap,
                    0,
                );
            }
        }, [inView]);

        return (
            <Box between={8} ref={ref}>
                {[...flightTariffs].map(([flightId, {title, tariffs}]) => {
                    const hasPlusRow = [...tariffs].some(([, tariff]) =>
                        isNotNull(
                            tariff.promoCampaigns?.plusPromo2021
                                ?.totalPlusPoints,
                        ),
                    );

                    const isSingleTariff = tariffs.size === 1;

                    return (
                        <Box key={flightId} between={5}>
                            <Box x="4">
                                <Heading
                                    level="2"
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'title',
                                    })}
                                >
                                    {title}
                                </Heading>
                            </Box>
                            <Box
                                className={cx('carousel', {
                                    carousel_single: isSingleTariff,
                                })}
                                inset={2}
                                innerRef={containerRef}
                            >
                                {[...tariffs].map(
                                    ([variantId, tariff], idx) => (
                                        <div
                                            key={variantId}
                                            className={cx('carousel__item')}
                                            ref={
                                                activeColumn === idx
                                                    ? activeCardRef
                                                    : undefined
                                            }
                                        >
                                            <FareTermsCard
                                                fareTerms={
                                                    tariff.fareTerms.terms
                                                }
                                                isActive={activeColumn === idx}
                                                price={tariff.priceInfo?.total}
                                                title={
                                                    tariff.fareTerms
                                                        .tariffGroupName
                                                }
                                                onSelect={(): void =>
                                                    onChange?.(variantId)
                                                }
                                                promoCampaigns={
                                                    tariff.promoCampaigns
                                                }
                                                hasPlusRow={hasPlusRow}
                                                maxWidth={isSingleTariff}
                                                {...prepareQaAttributes({
                                                    parent: props,
                                                    key: variantId,
                                                    current: 'tariff',
                                                })}
                                            />
                                        </div>
                                    ),
                                )}
                            </Box>
                        </Box>
                    );
                })}
            </Box>
        );
    };

interface IFareTermsCardProps extends IWithQaAttributes {
    isActive?: boolean;
    fareTerms: Record<string, ITermValue>;
    innerRef?: React.Ref<HTMLElement>;
    title: string;
    promoCampaigns?: TBookingPromoCampaigns;
    hasPlusRow?: boolean;
    price?: IPrice;
    maxWidth?: boolean;
    onSelect?(): void;
}

const FareTermsCard: React.FC<IFareTermsCardProps> = props => {
    const {
        isActive,
        title,
        fareTerms,
        innerRef,
        onSelect,
        price,
        promoCampaigns,
        hasPlusRow,
        maxWidth = false,
    } = props;

    const aviaAeroflotPlus = useSelector(aviaAeroflotPlusSelector);
    const plusPoints =
        aviaAeroflotPlus &&
        promoCampaigns?.plusPromo2021?.enabled &&
        promoCampaigns?.plusPromo2021?.totalPlusPoints
            ? promoCampaigns?.plusPromo2021?.totalPlusPoints
            : undefined;

    return (
        <Card
            shadow="default"
            x="3"
            y="4"
            innerRef={innerRef}
            className={cx(isActive && 'tariff-card_active')}
        >
            <FlightTariffTable>
                <FlightTariffTable.Row>
                    <FlightTariffTable.Cell className={cx({maxWidth})}>
                        <Flex
                            flexDirection="column"
                            alignItems="center"
                            below="4"
                        >
                            <Box
                                below="2"
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'name',
                                })}
                            >
                                {upperFirst(title.toLowerCase())}
                            </Box>

                            {price && (
                                <Box
                                    textSize="l"
                                    textWeight="medium"
                                    below={plusPoints ? 1 : 4}
                                >
                                    <Price
                                        {...price}
                                        {...prepareQaAttributes({
                                            parent: props,
                                            current: 'price',
                                        })}
                                    />
                                </Box>
                            )}
                            {plusPoints && (
                                <Box below="4">
                                    <TextWithIcon
                                        iconLeft={YandexPlusSignColorGradientS}
                                        iconLeftProps={{width: 16, height: 16}}
                                        spaceBetween="1"
                                        text={i18nBlockAviaPlus.bookingTariffTablePlusRefund(
                                            {
                                                count: plusPoints,
                                            },
                                        )}
                                    />
                                </Box>
                            )}
                            {onSelect && (
                                <Button
                                    className={cx({
                                        button_hasPlusRow:
                                            !plusPoints && hasPlusRow,
                                    })}
                                    size="l"
                                    theme="secondary"
                                    disabled={isActive}
                                    onClick={onSelect}
                                    width="max"
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'button',
                                    })}
                                >
                                    {isActive
                                        ? i18nBlock.current()
                                        : i18nBlock.select()}
                                </Button>
                            )}
                        </Flex>
                    </FlightTariffTable.Cell>
                </FlightTariffTable.Row>

                <FareTermRow
                    term={fareTerms.carryOn}
                    label={formatTermType('carryOn')}
                >
                    {(carryOn): ReactNode => (
                        <BaggageFormatter
                            {...carryOn}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'carryOn',
                            })}
                        />
                    )}
                </FareTermRow>

                <FareTermRow
                    term={fareTerms.baggage}
                    label={formatTermType('baggage')}
                >
                    {(baggage): ReactNode => (
                        <BaggageFormatter
                            showNoBaggage
                            places={baggage.places}
                            weight={baggage.weight}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'baggage',
                            })}
                        />
                    )}
                </FareTermRow>

                <FareTermRow
                    term={fareTerms.refundable}
                    label={formatTermType('refundable')}
                >
                    {(term): ReactNode => (
                        <FlightFareAvailabilityTerm
                            {...term}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'refundable',
                            })}
                        />
                    )}
                </FareTermRow>

                <FareTermRow
                    term={fareTerms.changingCarriage}
                    label={formatTermType('changingCarriage')}
                >
                    {(term): ReactNode => (
                        <FlightFareAvailabilityTerm
                            {...term}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'changingCarriage',
                            })}
                        />
                    )}
                </FareTermRow>

                <FareTermRow
                    term={fareTerms.miles}
                    label={formatTermType('miles')}
                    bottomBorder={false}
                >
                    {(miles): ReactNode => (
                        <span
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'miles',
                            })}
                        >
                            {miles.miles}
                        </span>
                    )}
                </FareTermRow>
            </FlightTariffTable>
        </Card>
    );
};
